const HEARTBEAT_INTERVAL = 5000;
const HEARTBEAT_TIMEOUT = 7000; // Should be > HEARTBEAT_INTERVAL
const MIN_REFRESH_INTERVAL = 5000; // Worker enforces this between actual API calls

let instanceId = null;
let isLeader = false;
let baseUrl = null;
let heartbeatIntervalId = null;
let leaderHeartbeatTimeoutId = null;
let isVisible = true;
let lastRefreshApiAttemptTime = 0;

const channel = new BroadcastChannel("token-refresh-coordination");

let isPerformingRefreshApiCall = false;
const pendingRefreshApiCallbacks = [];

function log(message) {
    postMessage({ type: "log", message: message });
}
function warn(message) {
    postMessage({ type: "warn", message: message });
}
function error(message) {
    postMessage({ type: "error", message: message });
}

function sendHeartbeat() {
    if (isLeader) {
        channel.postMessage({ type: "heartbeat", instanceId: instanceId });
    }
}

function resetLeaderTimeout() {
    if (leaderHeartbeatTimeoutId) clearTimeout(leaderHeartbeatTimeoutId);
    leaderHeartbeatTimeoutId = setTimeout(() => {
        log("No heartbeat received. Attempting leadership...");
        attemptLeadership();
    }, HEARTBEAT_TIMEOUT);
}

channel.onmessage = (event) => {
    const msg = event.data;
    if (msg.instanceId === instanceId) return;

    if (msg.type === "heartbeat") {
        if (!isLeader) {
            resetLeaderTimeout();
        }
    }

    if (msg.type === "requestLeadership") {
        if (isLeader) {
            sendHeartbeat();
        } else {
            resetLeaderTimeout();
        }
    }
};

function startHeartbeat() {
    if (heartbeatIntervalId) stopHeartbeat();
    sendHeartbeat();
    heartbeatIntervalId = setInterval(sendHeartbeat, HEARTBEAT_INTERVAL);
    log("Heartbeat started.");
}

function stopHeartbeat() {
    if (heartbeatIntervalId) {
        clearInterval(heartbeatIntervalId);
        heartbeatIntervalId = null;
        log("Heartbeat stopped.");
    }
}

async function getRefreshToken() {
    try {
        const refreshUrl = baseUrl + "/refresh";
        const response = await fetch(refreshUrl, {
            method: "POST",
            headers: { "Content-Type": "application/json" },
            credentials: "include"
        });

        if (!response.ok) {
             const errorBody = await response.text().catch(() => "N/A");
             throw new Error("HTTP error! status: " + response.status + " - " + response.statusText + ". Body: " + errorBody.substring(0, 200));
        }
        return response;
    } catch (err) {
        error("Error during raw fetch for refresh: " + err.message);
        throw err;
    }
}

async function fetchCurrentExpiryStatus() {
     try {
        log("Fetching current expiry status...");
        const response = await getRefreshToken();

        const data = await response.json();
        const expirySeconds =
            typeof data["expire-seconds"] === "number"
                ? data["expire-seconds"]
                : typeof data === "number"
                    ? data
                    : null;

        if (!expirySeconds || expirySeconds <= 0) {
            throw new Error("Invalid expiry format received from status endpoint: " + JSON.stringify(data));
        }

        log("Current expiry status fetched: " + expirySeconds + "s");
        postMessage({ type: "scheduleNextRefresh", expirySeconds: expirySeconds });

     } catch (err) {
        warn("Error fetching current expiry status: " + err.message);
        postMessage({ type: "scheduleFetchFailed", message: err.message });
     }
}

async function performQueuedRefresh() {
    if (isPerformingRefreshApiCall) {
        log("Refresh API call already in progress. Queuing request.");
        pendingRefreshApiCallbacks.push(() => performQueuedRefresh());
        return;
    }

    isPerformingRefreshApiCall = true;
    log("Starting queued refresh API call process.");

    const now = Date.now();
    const timeSinceLast = now - lastRefreshApiAttemptTime;
    if (timeSinceLast < MIN_REFRESH_INTERVAL) {
        const wait = MIN_REFRESH_INTERVAL - timeSinceLast;
        log("Minimum refresh interval not passed (" + timeSinceLast + "ms since last). Waiting " + wait + "ms before API call.");
        await new Promise(resolve => setTimeout(resolve, wait));
    }
    lastRefreshApiAttemptTime = Date.now();

    try {
        const response = await getRefreshToken();

        const data = await response.json();
        const newExpirySeconds =
            typeof data["expire-seconds"] === "number"
                ? data["expire-seconds"]
                : typeof data === "number"
                    ? data
                    : null;

        if (!newExpirySeconds || newExpirySeconds <= 0) {
            throw new Error("Invalid expiry format received after refresh: " + JSON.stringify(data));
        }

        log("Queued refresh API call successful.");
        postMessage({ type: "refreshSuccess", expirySeconds: newExpirySeconds });

    } catch (err) {
        error("Error during queued refresh API call: " + err.message);
        postMessage({ type: "refreshApiFailed", message: err.message });

    } finally {
        isPerformingRefreshApiCall = false;
        processPendingRefreshApiCallbacks();
    }
}

function processPendingRefreshApiCallbacks() {
    while (pendingRefreshApiCallbacks.length > 0 && !isPerformingRefreshApiCall) {
        const nextCallback = pendingRefreshApiCallbacks.shift();
        nextCallback();
    }
}

function attemptLeadership() {
    if (isLeader) return;
    log("Attempting to become leader...");
    isLeader = true;
    postMessage({ type: "isLeader", value: true });
    fetchCurrentExpiryStatus();
    startHeartbeat();
}

function stopBeingLeader() {
    if (!isLeader) return;
    isLeader = false;
    stopHeartbeat();
    postMessage({ type: "isLeader", value: false });
    log("Stopped being leader.");
}

function cleanup() {
    stopBeingLeader();
    channel.close();
    if (leaderHeartbeatTimeoutId) clearTimeout(leaderHeartbeatTimeoutId);
    log("Worker cleanup complete.");
}

onmessage = async function (event) {
    switch (event.data.type) {
        case "init":
            instanceId = event.data.instanceId;
            baseUrl = event.data.baseUrl;
            log("Worker initialized: " + instanceId + ", Base URL: " + baseUrl);
            channel.postMessage({ type: "requestLeadership", instanceId: instanceId });
            resetLeaderTimeout();
            break;

        case "visibilityChange":
            isVisible = event.data.isVisible;
            log("Visibility changed: " + isVisible);
            if (isLeader && isVisible) {
                 fetchCurrentExpiryStatus();
            } else if (isVisible && !isLeader) {
                 log("Visible but not leader. Requesting leadership...");
                 channel.postMessage({ type: "requestLeadership", instanceId: instanceId });
                 resetLeaderTimeout();
            }
            break;

        case "requestLeadership":
             if (!isLeader) {
                 log("Explicit leadership request received from main thread. Attempting leadership...");
                 attemptLeadership();
             } else {
                 sendHeartbeat();
             }
             break;

        case "fetchExpiry":
            if (isLeader) {
                fetchCurrentExpiryStatus();
            } else {
                warn("Received 'fetchExpiry' but not leader.");
            }
            break;

        case "performRefresh":
            if (isLeader) {
                performQueuedRefresh();
            } else {
                warn("Received 'performRefresh' but not leader.");
            }
            break;

        case "stopLeader":
            stopBeingLeader();
            break;

        case "cleanup":
            cleanup();
            break;
    }
};

onunload = function () {
    cleanup();
};