-- Migration to update variable uniqueness logic for soft delete strategy
-- This updates the variable registry functions and related triggers for soft delete
-- Note: deleted_dttm columns and unique indexes were already updated in V237 and V238

-- Add soft delete columns to variable_registry table (it doesn't have audit columns)
ALTER TABLE variable_registry ADD COLUMN deleted_dttm TIMESTAMP WITH TIME ZONE;
ALTER TABLE variable_registry ADD COLUMN deleted_user VARCHAR(50);

-- Update the unique constraint to work with soft delete
ALTER TABLE variable_registry DROP CONSTRAINT IF EXISTS uk_variable_registry_project_name;
CREATE UNIQUE INDEX uk_variable_registry_project_name_active
ON variable_registry (project_id, name)
WHERE deleted_dttm IS NULL;

-- Drop existing cleanup triggers since they're designed for hard deletes
DROP TRIGGER IF EXISTS tr_dnp3_variable_cleanup ON dnp3_variable;
DROP TRIGGER IF EXISTS tr_modbus_variable_cleanup ON modbus_variable;
DROP TRIGGER IF EXISTS tr_s7_variable_cleanup ON s7_variable;
DROP TRIGGER IF EXISTS tr_iec104_variable_cleanup ON iec104_variable;
DROP TRIGGER IF EXISTS tr_iec61850_variable_cleanup ON iec61850_variable;
DROP TRIGGER IF EXISTS tr_mqtt_variable_cleanup ON mqtt_variable;
DROP TRIGGER IF EXISTS tr_opc_ua_variable_cleanup ON opc_ua_variable;
DROP TRIGGER IF EXISTS tr_ethernet_ip_variable_cleanup ON ethernet_ip_variable;
DROP TRIGGER IF EXISTS tr_fatek_variable_cleanup ON fatek_variable;
DROP TRIGGER IF EXISTS tr_local_variable_cleanup ON local_variable;
DROP TRIGGER IF EXISTS tr_opc_da_variable_cleanup ON opc_da_variable;

-- Replace the cleanup function with a soft delete version
CREATE OR REPLACE FUNCTION soft_delete_variable_registry()
RETURNS TRIGGER AS $$
BEGIN
    -- For soft delete: mark variable as deleted in registry instead of removing it
    UPDATE inscada.variable_registry
    SET deleted_dttm = NEW.deleted_dttm,
        deleted_user = NEW.deleted_user
    WHERE variable_id = NEW.variable_id;

    RETURN NEW;
END;
$$ LANGUAGE plpgsql;

-- Update the validation function to check for soft delete status
CREATE OR REPLACE FUNCTION validate_variable_name_uniqueness()
RETURNS TRIGGER AS $$
BEGIN
    -- Check if variable name already exists in the same project (excluding soft-deleted and current variable)
    IF (TG_OP = 'INSERT' OR (TG_OP = 'UPDATE' AND OLD.name != NEW.name)) THEN
        IF EXISTS (
            SELECT 1 FROM inscada.variable_registry
            WHERE project_id = NEW.project_id
            AND name = NEW.name
            AND variable_id != NEW.variable_id
            AND deleted_dttm IS NULL  -- Only check active (non-deleted) variables
        ) THEN
            RAISE EXCEPTION 'Variable name "%" already exists in project "%"', NEW.name, NEW.project_id;
        END IF;
    END IF;

    -- For INSERT operations, add to variable_registry
    IF TG_OP = 'INSERT' THEN
        INSERT INTO inscada.variable_registry (variable_id, project_id, name, protocol, deleted_dttm, deleted_user)
        VALUES (NEW.variable_id, NEW.project_id, NEW.name,
            CASE TG_TABLE_NAME
                WHEN 'dnp3_variable' THEN 'DNP3'
                WHEN 'modbus_variable' THEN 'MODBUS'
                WHEN 's7_variable' THEN 'S7'
                WHEN 'iec104_variable' THEN 'IEC104'
                WHEN 'iec61850_variable' THEN 'IEC61850'
                WHEN 'mqtt_variable' THEN 'MQTT'
                WHEN 'opc_ua_variable' THEN 'OPC_UA'
                WHEN 'ethernet_ip_variable' THEN 'ETHERNET_IP'
                WHEN 'fatek_variable' THEN 'FATEK'
                WHEN 'local_variable' THEN 'LOCAL'
                WHEN 'opc_da_variable' THEN 'OPC_DA'
            END,
            NEW.deleted_dttm,  -- Set deleted_dttm from the variable record
            NEW.deleted_user   -- Set deleted_user from the variable record
        );
    END IF;

    -- For UPDATE operations, update variable_registry including soft delete status
    IF TG_OP = 'UPDATE' THEN
        UPDATE inscada.variable_registry
        SET project_id = NEW.project_id,
            name = NEW.name,
            deleted_dttm = NEW.deleted_dttm,  -- Update soft delete status
            deleted_user = NEW.deleted_user   -- Update who deleted it
        WHERE variable_id = NEW.variable_id;
    END IF;

    RETURN NEW;
END;
$$ LANGUAGE plpgsql;

-- Update the check_variable_exists function to exclude soft-deleted variables
CREATE OR REPLACE FUNCTION check_variable_exists(variable_id VARCHAR(36))
RETURNS BOOLEAN AS $$
BEGIN
    -- Empty string or NULL variable_id is considered as "exists"
    IF variable_id IS NULL OR variable_id = '' THEN
        RETURN TRUE;
    END IF;

    -- Check if variable exists in any of the protocol-specific tables (excluding soft-deleted)
    RETURN EXISTS (
        SELECT 1 FROM inscada.dnp3_variable v WHERE v.variable_id = $1 AND v.deleted_dttm IS NULL
        UNION
        SELECT 1 FROM inscada.modbus_variable v WHERE v.variable_id = $1 AND v.deleted_dttm IS NULL
        UNION
        SELECT 1 FROM inscada.s7_variable v WHERE v.variable_id = $1 AND v.deleted_dttm IS NULL
        UNION
        SELECT 1 FROM inscada.iec104_variable v WHERE v.variable_id = $1 AND v.deleted_dttm IS NULL
        UNION
        SELECT 1 FROM inscada.iec61850_variable v WHERE v.variable_id = $1 AND v.deleted_dttm IS NULL
        UNION
        SELECT 1 FROM inscada.mqtt_variable v WHERE v.variable_id = $1 AND v.deleted_dttm IS NULL
        UNION
        SELECT 1 FROM inscada.opc_ua_variable v WHERE v.variable_id = $1 AND v.deleted_dttm IS NULL
        UNION
        SELECT 1 FROM inscada.ethernet_ip_variable v WHERE v.variable_id = $1 AND v.deleted_dttm IS NULL
        UNION
        SELECT 1 FROM inscada.fatek_variable v WHERE v.variable_id = $1 AND v.deleted_dttm IS NULL
        UNION
        SELECT 1 FROM inscada.local_variable v WHERE v.variable_id = $1 AND v.deleted_dttm IS NULL
        UNION
        SELECT 1 FROM inscada.opc_da_variable v WHERE v.variable_id = $1 AND v.deleted_dttm IS NULL
    );
END;
$$ LANGUAGE plpgsql;

-- Update the is_variable_referenced function to exclude soft-deleted references
CREATE OR REPLACE FUNCTION is_variable_referenced(p_variable_id VARCHAR(36))
RETURNS BOOLEAN AS $$
BEGIN
    -- Return TRUE if the variable is referenced by any active (non-soft-deleted) table

    -- Check alarm table - off_time_variable_id
    IF EXISTS (
        SELECT 1 FROM information_schema.columns
        WHERE table_schema = 'inscada' AND table_name = 'alarm' AND column_name = 'off_time_variable_id'
    ) AND EXISTS (
        SELECT 1 FROM inscada.alarm WHERE off_time_variable_id = p_variable_id AND deleted_dttm IS NULL
    ) THEN
        RETURN TRUE;
    END IF;

    -- Check alarm table - on_time_variable_id
    IF EXISTS (
        SELECT 1 FROM information_schema.columns
        WHERE table_schema = 'inscada' AND table_name = 'alarm' AND column_name = 'on_time_variable_id'
    ) AND EXISTS (
        SELECT 1 FROM inscada.alarm WHERE on_time_variable_id = p_variable_id AND deleted_dttm IS NULL
    ) THEN
        RETURN TRUE;
    END IF;

    -- Check digital_alarm - variable_a_id
    IF EXISTS (
        SELECT 1 FROM information_schema.columns
        WHERE table_schema = 'inscada' AND table_name = 'digital_alarm' AND column_name = 'variable_a_id'
    ) AND EXISTS (
        SELECT 1 FROM inscada.digital_alarm WHERE variable_a_id = p_variable_id AND deleted_dttm IS NULL
    ) THEN
        RETURN TRUE;
    END IF;

    -- Check digital_alarm - variable_b_id
    IF EXISTS (
        SELECT 1 FROM information_schema.columns
        WHERE table_schema = 'inscada' AND table_name = 'digital_alarm' AND column_name = 'variable_b_id'
    ) AND EXISTS (
        SELECT 1 FROM inscada.digital_alarm WHERE variable_b_id = p_variable_id AND deleted_dttm IS NULL
    ) THEN
        RETURN TRUE;
    END IF;

    -- Check analog_alarm - variable_id
    IF EXISTS (
        SELECT 1 FROM information_schema.columns
        WHERE table_schema = 'inscada' AND table_name = 'analog_alarm' AND column_name = 'variable_id'
    ) AND EXISTS (
        SELECT 1 FROM inscada.analog_alarm WHERE variable_id = p_variable_id AND deleted_dttm IS NULL
    ) THEN
        RETURN TRUE;
    END IF;

    -- Check data_transfer_detail - source_var_id
    IF EXISTS (
        SELECT 1 FROM information_schema.columns
        WHERE table_schema = 'inscada' AND table_name = 'data_transfer_detail' AND column_name = 'source_var_id'
    ) AND EXISTS (
        SELECT 1 FROM inscada.data_transfer_detail WHERE source_var_id = p_variable_id AND deleted_dttm IS NULL
    ) THEN
        RETURN TRUE;
    END IF;

    -- Check data_transfer_detail - target_var_id
    IF EXISTS (
        SELECT 1 FROM information_schema.columns
        WHERE table_schema = 'inscada' AND table_name = 'data_transfer_detail' AND column_name = 'target_var_id'
    ) AND EXISTS (
        SELECT 1 FROM inscada.data_transfer_detail WHERE target_var_id = p_variable_id AND deleted_dttm IS NULL
    ) THEN
        RETURN TRUE;
    END IF;

    -- Check report_variable - deviation_variable_id
    IF EXISTS (
        SELECT 1 FROM information_schema.columns
        WHERE table_schema = 'inscada' AND table_name = 'report_variable' AND column_name = 'deviation_variable_id'
    ) AND EXISTS (
        SELECT 1 FROM inscada.report_variable WHERE deviation_variable_id = p_variable_id AND deleted_dttm IS NULL
    ) THEN
        RETURN TRUE;
    END IF;

    -- Check report_variable - total_variable_id
    IF EXISTS (
        SELECT 1 FROM information_schema.columns
        WHERE table_schema = 'inscada' AND table_name = 'report_variable' AND column_name = 'total_variable_id'
    ) AND EXISTS (
        SELECT 1 FROM inscada.report_variable WHERE total_variable_id = p_variable_id AND deleted_dttm IS NULL
    ) THEN
        RETURN TRUE;
    END IF;

    -- Check report_variable - variable_id
    IF EXISTS (
        SELECT 1 FROM information_schema.columns
        WHERE table_schema = 'inscada' AND table_name = 'report_variable' AND column_name = 'variable_id'
    ) AND EXISTS (
        SELECT 1 FROM inscada.report_variable WHERE variable_id = p_variable_id AND deleted_dttm IS NULL
    ) THEN
        RETURN TRUE;
    END IF;

    -- Check trend_tag - variable_id
    IF EXISTS (
        SELECT 1 FROM information_schema.columns
        WHERE table_schema = 'inscada' AND table_name = 'trend_tag' AND column_name = 'variable_id'
    ) AND EXISTS (
        SELECT 1 FROM inscada.trend_tag WHERE variable_id = p_variable_id AND deleted_dttm IS NULL
    ) THEN
        RETURN TRUE;
    END IF;

    -- Check map_variable - variable_id
    IF EXISTS (
        SELECT 1 FROM information_schema.columns
        WHERE table_schema = 'inscada' AND table_name = 'map_variable' AND column_name = 'variable_id'
    ) AND EXISTS (
        SELECT 1 FROM inscada.map_variable WHERE variable_id = p_variable_id AND deleted_dttm IS NULL
    ) THEN
        RETURN TRUE;
    END IF;

    -- Check monitor_variable - variable_id
    IF EXISTS (
        SELECT 1 FROM information_schema.columns
        WHERE table_schema = 'inscada' AND table_name = 'monitor_variable' AND column_name = 'variable_id'
    ) AND EXISTS (
        SELECT 1 FROM inscada.monitor_variable WHERE variable_id = p_variable_id AND deleted_dttm IS NULL
    ) THEN
        RETURN TRUE;
    END IF;

    -- Not referenced by any active (non-soft-deleted) table
    RETURN FALSE;
END;
$$ LANGUAGE plpgsql;

-- Create soft delete triggers for all 11 protocols
-- These triggers will update the registry when variables are soft deleted
CREATE TRIGGER tr_dnp3_variable_soft_delete
    AFTER UPDATE ON dnp3_variable
    FOR EACH ROW
    WHEN (OLD.deleted_dttm IS NULL AND NEW.deleted_dttm IS NOT NULL)
    EXECUTE FUNCTION soft_delete_variable_registry();

CREATE TRIGGER tr_modbus_variable_soft_delete
    AFTER UPDATE ON modbus_variable
    FOR EACH ROW
    WHEN (OLD.deleted_dttm IS NULL AND NEW.deleted_dttm IS NOT NULL)
    EXECUTE FUNCTION soft_delete_variable_registry();

CREATE TRIGGER tr_s7_variable_soft_delete
    AFTER UPDATE ON s7_variable
    FOR EACH ROW
    WHEN (OLD.deleted_dttm IS NULL AND NEW.deleted_dttm IS NOT NULL)
    EXECUTE FUNCTION soft_delete_variable_registry();

CREATE TRIGGER tr_iec104_variable_soft_delete
    AFTER UPDATE ON iec104_variable
    FOR EACH ROW
    WHEN (OLD.deleted_dttm IS NULL AND NEW.deleted_dttm IS NOT NULL)
    EXECUTE FUNCTION soft_delete_variable_registry();

CREATE TRIGGER tr_iec61850_variable_soft_delete
    AFTER UPDATE ON iec61850_variable
    FOR EACH ROW
    WHEN (OLD.deleted_dttm IS NULL AND NEW.deleted_dttm IS NOT NULL)
    EXECUTE FUNCTION soft_delete_variable_registry();

CREATE TRIGGER tr_mqtt_variable_soft_delete
    AFTER UPDATE ON mqtt_variable
    FOR EACH ROW
    WHEN (OLD.deleted_dttm IS NULL AND NEW.deleted_dttm IS NOT NULL)
    EXECUTE FUNCTION soft_delete_variable_registry();

CREATE TRIGGER tr_opc_ua_variable_soft_delete
    AFTER UPDATE ON opc_ua_variable
    FOR EACH ROW
    WHEN (OLD.deleted_dttm IS NULL AND NEW.deleted_dttm IS NOT NULL)
    EXECUTE FUNCTION soft_delete_variable_registry();

CREATE TRIGGER tr_ethernet_ip_variable_soft_delete
    AFTER UPDATE ON ethernet_ip_variable
    FOR EACH ROW
    WHEN (OLD.deleted_dttm IS NULL AND NEW.deleted_dttm IS NOT NULL)
    EXECUTE FUNCTION soft_delete_variable_registry();

CREATE TRIGGER tr_fatek_variable_soft_delete
    AFTER UPDATE ON fatek_variable
    FOR EACH ROW
    WHEN (OLD.deleted_dttm IS NULL AND NEW.deleted_dttm IS NOT NULL)
    EXECUTE FUNCTION soft_delete_variable_registry();

CREATE TRIGGER tr_local_variable_soft_delete
    AFTER UPDATE ON local_variable
    FOR EACH ROW
    WHEN (OLD.deleted_dttm IS NULL AND NEW.deleted_dttm IS NOT NULL)
    EXECUTE FUNCTION soft_delete_variable_registry();

CREATE TRIGGER tr_opc_da_variable_soft_delete
    AFTER UPDATE ON opc_da_variable
    FOR EACH ROW
    WHEN (OLD.deleted_dttm IS NULL AND NEW.deleted_dttm IS NOT NULL)
    EXECUTE FUNCTION soft_delete_variable_registry();

-- Clean up existing variable_registry data to mark soft-deleted variables
-- Update registry to reflect current soft delete status of variables
UPDATE inscada.variable_registry vr
SET deleted_dttm = var.deleted_dttm,
    deleted_user = var.deleted_user
FROM (
    SELECT variable_id, deleted_dttm, deleted_user FROM dnp3_variable WHERE deleted_dttm IS NOT NULL
    UNION ALL
    SELECT variable_id, deleted_dttm, deleted_user FROM modbus_variable WHERE deleted_dttm IS NOT NULL
    UNION ALL
    SELECT variable_id, deleted_dttm, deleted_user FROM s7_variable WHERE deleted_dttm IS NOT NULL
    UNION ALL
    SELECT variable_id, deleted_dttm, deleted_user FROM iec104_variable WHERE deleted_dttm IS NOT NULL
    UNION ALL
    SELECT variable_id, deleted_dttm, deleted_user FROM iec61850_variable WHERE deleted_dttm IS NOT NULL
    UNION ALL
    SELECT variable_id, deleted_dttm, deleted_user FROM mqtt_variable WHERE deleted_dttm IS NOT NULL
    UNION ALL
    SELECT variable_id, deleted_dttm, deleted_user FROM opc_ua_variable WHERE deleted_dttm IS NOT NULL
    UNION ALL
    SELECT variable_id, deleted_dttm, deleted_user FROM ethernet_ip_variable WHERE deleted_dttm IS NOT NULL
    UNION ALL
    SELECT variable_id, deleted_dttm, deleted_user FROM fatek_variable WHERE deleted_dttm IS NOT NULL
    UNION ALL
    SELECT variable_id, deleted_dttm, deleted_user FROM local_variable WHERE deleted_dttm IS NOT NULL
    UNION ALL
    SELECT variable_id, deleted_dttm, deleted_user FROM opc_da_variable WHERE deleted_dttm IS NOT NULL
) var
WHERE vr.variable_id = var.variable_id;
