-- Migration to update connection uniqueness logic for soft delete strategy
-- This updates the cleanup_connection_registry function and related triggers
-- Note: deleted_dttm columns and unique indexes were already updated in V237 and V238
-- However, connection_registry does not have audit columns, so we need to add deleted_dttm manually

-- Add deleted_dttm column to connection_registry table (it doesn't have audit columns)
ALTER TABLE connection_registry ADD COLUMN deleted_dttm TIMESTAMP WITH TIME ZONE;
ALTER TABLE connection_registry ADD COLUMN deleted_user VARCHAR(50);

-- Update the unique constraint to work with soft delete
ALTER TABLE connection_registry DROP CONSTRAINT IF EXISTS uk_connection_registry_project_name;
CREATE UNIQUE INDEX uk_connection_registry_project_name_active
ON connection_registry (project_id, name)
WHERE deleted_dttm IS NULL;

-- Drop existing cleanup triggers since they're designed for hard deletes
DROP TRIGGER IF EXISTS tr_dnp3_connection_cleanup ON dnp3_connection;
DROP TRIGGER IF EXISTS tr_modbus_connection_cleanup ON modbus_connection;
DROP TRIGGER IF EXISTS tr_s7_connection_cleanup ON s7_connection;
DROP TRIGGER IF EXISTS tr_iec104_connection_cleanup ON iec104_connection;
DROP TRIGGER IF EXISTS tr_iec61850_connection_cleanup ON iec61850_connection;
DROP TRIGGER IF EXISTS tr_mqtt_connection_cleanup ON mqtt_connection;
DROP TRIGGER IF EXISTS tr_opc_ua_connection_cleanup ON opc_ua_connection;
DROP TRIGGER IF EXISTS tr_ethernet_ip_connection_cleanup ON ethernet_ip_connection;
DROP TRIGGER IF EXISTS tr_fatek_connection_cleanup ON fatek_connection;
DROP TRIGGER IF EXISTS tr_local_connection_cleanup ON local_connection;
DROP TRIGGER IF EXISTS tr_opc_da_connection_cleanup ON opc_da_connection;

-- Replace the cleanup function with a soft delete version
CREATE OR REPLACE FUNCTION soft_delete_connection_registry()
RETURNS TRIGGER AS $$
BEGIN
    -- For soft delete: mark connection as deleted in registry instead of removing it
    UPDATE inscada.connection_registry
    SET deleted_dttm = NEW.deleted_dttm,
        deleted_user = NEW.deleted_user
    WHERE conn_id = NEW.conn_id;

    RETURN NEW;
END;
$$ LANGUAGE plpgsql;

-- Update the validation function to check for soft delete status
CREATE OR REPLACE FUNCTION validate_connection_name_uniqueness()
RETURNS TRIGGER AS $$
BEGIN
    -- Check if connection name already exists in the same project (excluding soft-deleted and current connection)
    IF (TG_OP = 'INSERT' OR (TG_OP = 'UPDATE' AND (OLD.name != NEW.name OR OLD.project_id != NEW.project_id))) THEN
        IF EXISTS (
            SELECT 1 FROM inscada.connection_registry
            WHERE project_id = NEW.project_id
            AND name = NEW.name
            AND conn_id != NEW.conn_id
            AND deleted_dttm IS NULL  -- Only check active (non-deleted) connections
        ) THEN
            RAISE EXCEPTION 'Connection name "%" already exists in project "%"', NEW.name, NEW.project_id;
        END IF;
    END IF;

    -- For INSERT operations, add to connection_registry
    IF TG_OP = 'INSERT' THEN
        INSERT INTO inscada.connection_registry (conn_id, project_id, name, protocol, deleted_dttm, deleted_user)
        VALUES (NEW.conn_id, NEW.project_id, NEW.name,
            CASE TG_TABLE_NAME
                WHEN 'dnp3_connection' THEN 'DNP3'
                WHEN 'modbus_connection' THEN 'MODBUS'
                WHEN 's7_connection' THEN 'S7'
                WHEN 'iec104_connection' THEN 'IEC104'
                WHEN 'iec61850_connection' THEN 'IEC61850'
                WHEN 'mqtt_connection' THEN 'MQTT'
                WHEN 'opc_ua_connection' THEN 'OPC_UA'
                WHEN 'ethernet_ip_connection' THEN 'ETHERNET_IP'
                WHEN 'fatek_connection' THEN 'FATEK'
                WHEN 'local_connection' THEN 'LOCAL'
                WHEN 'opc_da_connection' THEN 'OPC_DA'
            END,
            NEW.deleted_dttm,  -- Set deleted_dttm from the connection record
            NEW.deleted_user   -- Set deleted_user from the connection record
        );
    END IF;

    -- For UPDATE operations, update connection_registry including soft delete status
    IF TG_OP = 'UPDATE' THEN
        UPDATE inscada.connection_registry
        SET project_id = NEW.project_id,
            name = NEW.name,
            deleted_dttm = NEW.deleted_dttm,  -- Update soft delete status
            deleted_user = NEW.deleted_user   -- Update who deleted it
        WHERE conn_id = NEW.conn_id;
    END IF;

    RETURN NEW;
END;
$$ LANGUAGE plpgsql;

-- Create soft delete triggers for all 11 protocols
-- These triggers will update the registry when connections are soft deleted
CREATE TRIGGER tr_dnp3_connection_soft_delete
    AFTER UPDATE ON dnp3_connection
    FOR EACH ROW
    WHEN (OLD.deleted_dttm IS NULL AND NEW.deleted_dttm IS NOT NULL)
    EXECUTE FUNCTION soft_delete_connection_registry();

CREATE TRIGGER tr_modbus_connection_soft_delete
    AFTER UPDATE ON modbus_connection
    FOR EACH ROW
    WHEN (OLD.deleted_dttm IS NULL AND NEW.deleted_dttm IS NOT NULL)
    EXECUTE FUNCTION soft_delete_connection_registry();

CREATE TRIGGER tr_s7_connection_soft_delete
    AFTER UPDATE ON s7_connection
    FOR EACH ROW
    WHEN (OLD.deleted_dttm IS NULL AND NEW.deleted_dttm IS NOT NULL)
    EXECUTE FUNCTION soft_delete_connection_registry();

CREATE TRIGGER tr_iec104_connection_soft_delete
    AFTER UPDATE ON iec104_connection
    FOR EACH ROW
    WHEN (OLD.deleted_dttm IS NULL AND NEW.deleted_dttm IS NOT NULL)
    EXECUTE FUNCTION soft_delete_connection_registry();

CREATE TRIGGER tr_iec61850_connection_soft_delete
    AFTER UPDATE ON iec61850_connection
    FOR EACH ROW
    WHEN (OLD.deleted_dttm IS NULL AND NEW.deleted_dttm IS NOT NULL)
    EXECUTE FUNCTION soft_delete_connection_registry();

CREATE TRIGGER tr_mqtt_connection_soft_delete
    AFTER UPDATE ON mqtt_connection
    FOR EACH ROW
    WHEN (OLD.deleted_dttm IS NULL AND NEW.deleted_dttm IS NOT NULL)
    EXECUTE FUNCTION soft_delete_connection_registry();

CREATE TRIGGER tr_opc_ua_connection_soft_delete
    AFTER UPDATE ON opc_ua_connection
    FOR EACH ROW
    WHEN (OLD.deleted_dttm IS NULL AND NEW.deleted_dttm IS NOT NULL)
    EXECUTE FUNCTION soft_delete_connection_registry();

CREATE TRIGGER tr_ethernet_ip_connection_soft_delete
    AFTER UPDATE ON ethernet_ip_connection
    FOR EACH ROW
    WHEN (OLD.deleted_dttm IS NULL AND NEW.deleted_dttm IS NOT NULL)
    EXECUTE FUNCTION soft_delete_connection_registry();

CREATE TRIGGER tr_fatek_connection_soft_delete
    AFTER UPDATE ON fatek_connection
    FOR EACH ROW
    WHEN (OLD.deleted_dttm IS NULL AND NEW.deleted_dttm IS NOT NULL)
    EXECUTE FUNCTION soft_delete_connection_registry();

CREATE TRIGGER tr_local_connection_soft_delete
    AFTER UPDATE ON local_connection
    FOR EACH ROW
    WHEN (OLD.deleted_dttm IS NULL AND NEW.deleted_dttm IS NOT NULL)
    EXECUTE FUNCTION soft_delete_connection_registry();

CREATE TRIGGER tr_opc_da_connection_soft_delete
    AFTER UPDATE ON opc_da_connection
    FOR EACH ROW
    WHEN (OLD.deleted_dttm IS NULL AND NEW.deleted_dttm IS NOT NULL)
    EXECUTE FUNCTION soft_delete_connection_registry();

-- Clean up existing connection_registry data to mark soft-deleted connections
-- Update registry to reflect current soft delete status of connections
UPDATE inscada.connection_registry cr
SET deleted_dttm = conn.deleted_dttm,
    deleted_user = conn.deleted_user
FROM (
    SELECT conn_id, deleted_dttm, deleted_user FROM dnp3_connection WHERE deleted_dttm IS NOT NULL
    UNION ALL
    SELECT conn_id, deleted_dttm, deleted_user FROM modbus_connection WHERE deleted_dttm IS NOT NULL
    UNION ALL
    SELECT conn_id, deleted_dttm, deleted_user FROM s7_connection WHERE deleted_dttm IS NOT NULL
    UNION ALL
    SELECT conn_id, deleted_dttm, deleted_user FROM iec104_connection WHERE deleted_dttm IS NOT NULL
    UNION ALL
    SELECT conn_id, deleted_dttm, deleted_user FROM iec61850_connection WHERE deleted_dttm IS NOT NULL
    UNION ALL
    SELECT conn_id, deleted_dttm, deleted_user FROM mqtt_connection WHERE deleted_dttm IS NOT NULL
    UNION ALL
    SELECT conn_id, deleted_dttm, deleted_user FROM opc_ua_connection WHERE deleted_dttm IS NOT NULL
    UNION ALL
    SELECT conn_id, deleted_dttm, deleted_user FROM ethernet_ip_connection WHERE deleted_dttm IS NOT NULL
    UNION ALL
    SELECT conn_id, deleted_dttm, deleted_user FROM fatek_connection WHERE deleted_dttm IS NOT NULL
    UNION ALL
    SELECT conn_id, deleted_dttm, deleted_user FROM local_connection WHERE deleted_dttm IS NOT NULL
    UNION ALL
    SELECT conn_id, deleted_dttm, deleted_user FROM opc_da_connection WHERE deleted_dttm IS NOT NULL
) conn
WHERE cr.conn_id = conn.conn_id;
