const HEARTBEAT_INTERVAL = 5000; // 5 seconds
const HEARTBEAT_TIMEOUT = 7000;   // 7 seconds (must be > HEARTBEAT_INTERVAL)
const MIN_REFRESH_INTERVAL = 5000; // 5 seconds minimum delay between refresh API calls

let instanceId = null;
let isLeader = false;
let baseUrl = null;
let heartbeatIntervalId = null;
let leaderHeartbeatTimeoutId = null;
let isVisible = true; // Tracks main thread visibility
let lastRefreshApiAttemptTime = 0;

const channel = new BroadcastChannel("token-refresh-coordination");

// Refresh coordination (centralized in worker)
let refreshInProgress = false; // True if getRefreshToken() is actively making a fetch call
let refreshWaiters = [];       // Callbacks waiting for the current refresh to complete

function log(msg) {
    postMessage({ type: "log", message: `[Worker-${instanceId}] ${msg}` });
}
function warn(msg) {
    postMessage({ type: "warn", message: `[Worker-${instanceId}] ${msg}` });
}
function error(msg) {
    postMessage({ type: "error", message: `[Worker-${instanceId}] ${msg}` });
}

function sendHeartbeat() {
    if (isLeader) {
        channel.postMessage({ type: "heartbeat", instanceId });
        log(`Sent heartbeat.`);
    }
}

function resetLeaderTimeout() {
    if (leaderHeartbeatTimeoutId) {
        clearTimeout(leaderHeartbeatTimeoutId);
    }
    leaderHeartbeatTimeoutId = setTimeout(() => {
        log(`Leader heartbeat timeout. Attempting leadership.`);
        attemptLeadership(); // Leader hasn't sent a heartbeat for too long, try to become leader
    }, HEARTBEAT_TIMEOUT);
    log(`Reset leader timeout.`);
}

channel.onmessage = (event) => {
    const msg = event.data;
    if (msg.instanceId === instanceId) {
        // Ignore messages from self
        // log(`Received message from self: ${msg.type}`);
        return;
    }
    log(`Received broadcast from another instance: Type=${msg.type}, Instance=${msg.instanceId}`);

    switch (msg.type) {
        case "heartbeat":
            // Any instance receiving a heartbeat from another instance should reset its timeout
            // regardless of whether it thinks it's the leader or not.
            // This ensures that if a new leader emerged, others acknowledge it.
            resetLeaderTimeout();
            if (isLeader) {
                // If I'm leader and received heartbeat from another, potential conflict or new tab opening.
                // Re-assert leadership or gracefully step down if instanceId is numerically lower/higher.
                // For simplicity, current leader remains leader as long as it's active.
                // More robust leader election could involve comparing instanceId.
                // For now, if I'm leader, I just keep sending my own heartbeats.
                log(`Received heartbeat from another leader candidate.`);
            }
            break;
        case "requestLeadership":
            if (isLeader) {
                log(`Received requestLeadership from ${msg.instanceId}. Sending my heartbeat.`);
                sendHeartbeat(); // Re-assert leadership
            } else {
                log(`Received requestLeadership from ${msg.instanceId}. Resetting my timeout (deferring to potential existing leader).`);
                resetLeaderTimeout(); // Acknowledge that another tab is active and might be leader
            }
            break;
        case "refreshing":
            if (!refreshInProgress) { // Only set if not already in progress
                refreshInProgress = true;
                log(`Received 'refreshing' signal from another instance.`);
            }
            break;
        case "refreshDone":
            refreshInProgress = false;
            log(`Received 'refreshDone' signal from another instance. Result: ${msg.result ? 'success' : 'failure'}`);
            // Resolve all waiters with the result from the leader's refresh
            const result = msg.result ? new Response(JSON.stringify(msg.result), { status: 200, headers: { 'Content-Type': 'application/json' } }) : null; // Reconstruct Response if needed by waiters
            refreshWaiters.forEach(cb => cb(result));
            refreshWaiters = [];
            break;
    }
};

function startHeartbeat() {
    if (heartbeatIntervalId) {
        stopHeartbeat();
    }
    sendHeartbeat(); // Send immediately on start
    heartbeatIntervalId = setInterval(sendHeartbeat, HEARTBEAT_INTERVAL);
    log(`Started heartbeat interval.`);
}

function stopHeartbeat() {
    if (heartbeatIntervalId) {
        clearInterval(heartbeatIntervalId);
        heartbeatIntervalId = null;
        log(`Stopped heartbeat interval.`);
    }
}

async function getRefreshToken() {
    // This function acts as the single point of truth for making the refresh API call.
    // It handles debouncing and queuing across all worker-side calls.
    if (refreshInProgress) {
        log(`Refresh already in progress. Waiting for result.`);
        return new Promise(resolve => {
            refreshWaiters.push(resolve);
        });
    }

    refreshInProgress = true;
    channel.postMessage({ type: "refreshing", instanceId }); // Inform other tabs a refresh is starting
    log(`Initiating refresh API call.`);

    const now = Date.now();
    const timeSinceLast = now - lastRefreshApiAttemptTime;
    if (timeSinceLast < MIN_REFRESH_INTERVAL) {
        const waitTime = MIN_REFRESH_INTERVAL - timeSinceLast;
        log(`Debouncing refresh API call for ${waitTime}ms.`);
        await new Promise(resolve => setTimeout(resolve, waitTime));
    }
    lastRefreshApiAttemptTime = Date.now(); // Update timestamp after potential wait

    try {
        const response = await fetch(baseUrl + "/refresh", {
            method: "POST",
            headers: { "Content-Type": "application/json" },
            credentials: "include" // Important for sending cookies
        });

        if (!response.ok) {
            error(`Refresh API call failed with status: ${response.status}`);
            if (response.status === 401) {
                postMessage({ type: "sessionExpired" });
                log(`Session expired, signaling main thread.`);
            }
            // Signal others that refresh is done (with failure)
            channel.postMessage({ type: "refreshDone", instanceId, result: null });
            throw new Error(`HTTP ${response.status} - ${response.statusText}`);
        }

        const data = await response.json(); // Read JSON here to pass it on
        log(`Refresh API call successful.`);

        // Signal others that refresh is done (with success)
        channel.postMessage({ type: "refreshDone", instanceId, result: data });
        return new Response(JSON.stringify(data), { status: response.status, headers: { 'Content-Type': 'application/json' } }); // Return a new Response object if original is consumed
    } catch (err) {
        error(`Refresh API call threw an error: ${err.message}`);
        channel.postMessage({ type: "refreshDone", instanceId, result: null }); // Signal others failure
        throw err;
    } finally {
        refreshInProgress = false; // Ensure flag is reset
        // No need to process pendingRefreshApiCallbacks here; getRefreshToken handles its own queueing.
    }
}

async function fetchCurrentExpiryStatus() {
    if (!isLeader) {
        log(`Not leader, skipping fetchCurrentExpiryStatus.`);
        return;
    }
    log(`Leader attempting to fetch current expiry status.`);
    try {
        const response = await getRefreshToken(); // Use the shared refresh function
        const data = await response.json(); // Data should be available from getRefreshToken
        const expirySeconds =
            typeof data["expire-seconds"] === "number"
                ? data["expire-seconds"]
                : typeof data === "number"
                    ? data
                    : null;

        if (!expirySeconds || expirySeconds <= 0) {
            throw new Error("Invalid expiry seconds received from API.");
        }
        log(`Fetched expiry: ${expirySeconds} seconds.`);
        postMessage({ type: "scheduleNextRefresh", expirySeconds });
    } catch (err) {
        error(`Failed to fetch expiry status: ${err.message}`);
        postMessage({ type: "scheduleFetchFailed", message: err.message });
    }
}

async function performQueuedRefresh() {
    if (!isLeader) {
        log(`Not leader, skipping performQueuedRefresh.`);
        return;
    }
    log(`Leader attempting to perform queued refresh.`);
    try {
        const response = await getRefreshToken(); // Use the shared refresh function
        const data = await response.json(); // Data should be available from getRefreshToken
        const newExpirySeconds =
            typeof data["expire-seconds"] === "number"
                ? data["expire-seconds"]
                : typeof data === "number"
                    ? data
                    : null;

        if (!newExpirySeconds || newExpirySeconds <= 0) {
            throw new Error("Invalid new expiry seconds received from API.");
        }
        log(`Refresh successful. New expiry: ${newExpirySeconds} seconds.`);
        postMessage({ type: "refreshSuccess", expirySeconds: newExpirySeconds });
    } catch (err) {
        error(`Failed to perform queued refresh: ${err.message}`);
        postMessage({ type: "refreshApiFailed", message: err.message });
    }
}

function attemptLeadership() {
    if (isLeader) {
        log(`Already leader, skipping attemptLeadership.`);
        return;
    }
    isLeader = true;
    log(`Attempting leadership. Declared self as leader.`);
    postMessage({ type: "isLeader", value: true });
    // After becoming leader, immediately fetch expiry to set up refresh schedule
    fetchCurrentExpiryStatus();
    startHeartbeat(); // Start sending heartbeats
    resetLeaderTimeout(); // Reset own timeout to manage self-monitoring as leader
}

function stopBeingLeader() {
    if (!isLeader) {
        log(`Already not leader, skipping stopBeingLeader.`);
        return;
    }
    isLeader = false;
    stopHeartbeat();
    if (leaderHeartbeatTimeoutId) {
        clearTimeout(leaderHeartbeatTimeoutId); // Clear the timeout as we are no longer monitoring for leader
        leaderHeartbeatTimeoutId = null;
    }
    log(`Stopped being leader.`);
    postMessage({ type: "isLeader", value: false });
}

function cleanup() {
    log(`Performing worker cleanup.`);
    stopBeingLeader(); // Ensure all leadership-related intervals/timeouts are cleared
    if (channel) {
        channel.close(); // Close the BroadcastChannel
        log(`BroadcastChannel closed.`);
    }
}

// Main thread communications
onmessage = async function (event) {
    log(`Received message from main thread: Type=${event.data.type}`);
    switch (event.data.type) {
        case "init":
            instanceId = event.data.instanceId;
            baseUrl = event.data.baseUrl;
            log(`Worker initialized with instance ID: ${instanceId}, Base URL: ${baseUrl}`);
            // When initialized, immediately request leadership to kick off election
            channel.postMessage({ type: "requestLeadership", instanceId });
            resetLeaderTimeout(); // Start monitoring for a leader (or self-elect)
            break;
        case "visibilityChange":
            isVisible = event.data.isVisible;
            log(`Visibility changed to: ${isVisible}`);
            break;
        case "requestLeadership":
            // If another instance requests leadership, respond if we are the leader
            if (isLeader) {
                sendHeartbeat();
            } else {
                // If not leader, just reset timeout to ensure we are still listening for a leader
                resetLeaderTimeout();
            }
            break;
        case "fetchExpiry":
            if (isLeader) {
                fetchCurrentExpiryStatus();
            } else {
                warn(`Received fetchExpiry command but not leader.`);
            }
            break;
        case "performRefresh":
            if (isLeader) {
                performQueuedRefresh();
            } else {
                warn(`Received performRefresh command but not leader.`);
            }
            break;
        case "stopLeader":
            stopBeingLeader(); // Main thread explicitly told us to stop leadership
            break;
        case "cleanup":
            cleanup();
            break;
    }
};

// On unload event (not strictly standard for workers, but good practice if supported/emulated)
// For Web Workers, the lifecycle is tied to the main thread's page lifecycle.
// When the page closes, the worker is terminated. `onunload` is not usually triggered directly in workers.
// The `cleanup` message from the main thread's `beforeunload` is the primary mechanism.
// However, including it doesn't hurt.
onunload = function () {
    log(`Worker unload detected.`);
    cleanup();
};