-- First add new columns to base frame table
ALTER TABLE frame
ADD COLUMN scan_time_factor integer,
ADD COLUMN readable_flag boolean NOT NULL DEFAULT true,
ADD COLUMN writable_flag boolean NOT NULL DEFAULT true;

-- Create a function to check if column exists
CREATE OR REPLACE FUNCTION column_exists(tname text, cname text)
RETURNS boolean AS $$
BEGIN
    RETURN EXISTS (
        SELECT 1
        FROM information_schema.columns
        WHERE table_name = tname
        AND column_name = cname
    );
END;
$$ LANGUAGE plpgsql;

-- Migrate data from each derived table if columns exist
DO $$
DECLARE
    derived_tables text[] := ARRAY[
        'dnp3_frame', 'ethernet_ip_frame', 'fatek_frame',
        'iec104_frame', 'iec61850_frame', 'local_frame',
        'modbus_frame', 'mqtt_frame', 'opc_da_frame',
        'opc_ua_frame', 's7_frame'
    ];
    t text;
BEGIN
    FOREACH t IN ARRAY derived_tables LOOP
        -- Migrate scan_time_factor if exists
        IF column_exists(t, 'scan_time_factor') THEN
            EXECUTE format('
                UPDATE frame f
                SET scan_time_factor = df.scan_time_factor
                FROM %I df
                WHERE f.frame_id = df.frame_id', t);
        END IF;

        -- Migrate readable_flag if exists
        IF column_exists(t, 'readable_flag') THEN
            EXECUTE format('
                UPDATE frame f
                SET readable_flag = df.readable_flag
                FROM %I df
                WHERE f.frame_id = df.frame_id', t);
        END IF;

        -- Migrate writable_flag if exists
        IF column_exists(t, 'writable_flag') THEN
            EXECUTE format('
                UPDATE frame f
                SET writable_flag = df.writable_flag
                FROM %I df
                WHERE f.frame_id = df.frame_id', t);
        END IF;

        -- Drop columns if they exist
        IF column_exists(t, 'scan_time_factor') THEN
            EXECUTE format('ALTER TABLE %I DROP COLUMN scan_time_factor', t);
        END IF;
        IF column_exists(t, 'readable_flag') THEN
            EXECUTE format('ALTER TABLE %I DROP COLUMN readable_flag', t);
        END IF;
        IF column_exists(t, 'writable_flag') THEN
            EXECUTE format('ALTER TABLE %I DROP COLUMN writable_flag', t);
        END IF;
    END LOOP;
END $$;

-- Clean up
DROP FUNCTION column_exists;

ALTER TABLE opc_da_frame DROP COLUMN IF EXISTS active_flag;

-- Add scan_type column to device tables
ALTER TABLE opc_ua_device 
ADD COLUMN scan_type varchar(20) NOT NULL DEFAULT 'Periodic';

ALTER TABLE opc_da_device 
ADD COLUMN scan_type varchar(20) NOT NULL DEFAULT 'Periodic';

ALTER TABLE local_device 
ADD COLUMN scan_type varchar(20) NOT NULL DEFAULT 'Periodic';

ALTER TABLE iec104_device 
ADD COLUMN scan_type varchar(20) NOT NULL DEFAULT 'Periodic';

ALTER TABLE iec61850_device 
ADD COLUMN scan_type varchar(20) NOT NULL DEFAULT 'Periodic';

ALTER TABLE dnp3_device 
ADD COLUMN scan_type varchar(20) NOT NULL DEFAULT 'Periodic';