-- Dynamic script to add soft delete columns to all tables with audit fields
-- This script queries the information_schema to find tables with audit columns
-- and generates ALTER TABLE statements for adding deleted_dttm and deleted_user

DO $$
DECLARE
    table_record RECORD;
    sql_statement TEXT;
BEGIN
    -- Find all tables in inscada schema that have both insert_user and insert_dttm columns
    FOR table_record IN
        SELECT DISTINCT t.table_name
        FROM information_schema.tables t
        WHERE t.table_schema = 'inscada'
          AND t.table_type = 'BASE TABLE'
          AND EXISTS (
              SELECT 1
              FROM information_schema.columns c1
              WHERE c1.table_schema = 'inscada'
                AND c1.table_name = t.table_name
                AND c1.column_name = 'insert_user'
          )
          AND EXISTS (
              SELECT 1
              FROM information_schema.columns c2
              WHERE c2.table_schema = 'inscada'
                AND c2.table_name = t.table_name
                AND c2.column_name = 'insert_dttm'
          )
          AND NOT EXISTS (
              SELECT 1
              FROM information_schema.columns c3
              WHERE c3.table_schema = 'inscada'
                AND c3.table_name = t.table_name
                AND c3.column_name = 'deleted_dttm'
          )
        ORDER BY t.table_name
    LOOP
        -- Add deleted_dttm column
        sql_statement := 'ALTER TABLE inscada.' || table_record.table_name || ' ADD COLUMN deleted_dttm TIMESTAMP WITH TIME ZONE;';
        EXECUTE sql_statement;
        RAISE NOTICE 'Added deleted_dttm to table: %', table_record.table_name;

        -- Add deleted_user column
        sql_statement := 'ALTER TABLE inscada.' || table_record.table_name || ' ADD COLUMN deleted_user VARCHAR(50);';
        EXECUTE sql_statement;
        RAISE NOTICE 'Added deleted_user to table: %', table_record.table_name;

        -- Create performance index
        sql_statement := 'CREATE INDEX idx_' || table_record.table_name || '_deleted_dttm ON inscada.' || table_record.table_name || '(deleted_dttm) WHERE deleted_dttm IS NULL;';
        EXECUTE sql_statement;
        RAISE NOTICE 'Created index for table: %', table_record.table_name;
    END LOOP;

    RAISE NOTICE 'Soft delete columns added to all tables with audit fields in inscada schema';
END $$;
