-- =============================================================================
-- VARIABLE UNIQUENESS
-- =============================================================================

CREATE OR REPLACE FUNCTION check_variable_exists(variable_id VARCHAR(36))
RETURNS BOOLEAN AS $$
BEGIN
    -- Empty string or NULL variable_id is considered as "exists"
    IF variable_id IS NULL OR variable_id = '' THEN
        RETURN TRUE;
    END IF;

    -- Check if variable exists in any of the protocol-specific tables
    RETURN EXISTS (
        SELECT 1 FROM inscada.dnp3_variable v WHERE v.variable_id = $1
        UNION
        SELECT 1 FROM inscada.modbus_variable v WHERE v.variable_id = $1
        UNION
        SELECT 1 FROM inscada.s7_variable v WHERE v.variable_id = $1
        UNION
        SELECT 1 FROM inscada.iec104_variable v WHERE v.variable_id = $1
        UNION
        SELECT 1 FROM inscada.iec61850_variable v WHERE v.variable_id = $1
        UNION
        SELECT 1 FROM inscada.mqtt_variable v WHERE v.variable_id = $1
        UNION
        SELECT 1 FROM inscada.opc_ua_variable v WHERE v.variable_id = $1
        UNION
        SELECT 1 FROM inscada.ethernet_ip_variable v WHERE v.variable_id = $1
        UNION
        SELECT 1 FROM inscada.fatek_variable v WHERE v.variable_id = $1
        UNION
        SELECT 1 FROM inscada.local_variable v WHERE v.variable_id = $1
        UNION
        SELECT 1 FROM inscada.opc_da_variable v WHERE v.variable_id = $1
    );
END;
$$ LANGUAGE plpgsql;

-- Trigger function to validate variable references
CREATE OR REPLACE FUNCTION validate_variable_reference()
RETURNS TRIGGER AS $$
DECLARE
    column_exists BOOLEAN;
BEGIN
    -- For alarm table with off_time_variable_id
    IF TG_TABLE_NAME = 'alarm' THEN
        SELECT EXISTS (
            SELECT 1 FROM information_schema.columns
            WHERE table_schema = 'inscada' AND table_name = 'alarm' AND column_name = 'off_time_variable_id'
        ) INTO column_exists;

        IF column_exists AND NEW.off_time_variable_id IS NOT NULL AND NOT inscada.check_variable_exists(NEW.off_time_variable_id) THEN
            RAISE EXCEPTION 'Variable with id % does not exist', NEW.off_time_variable_id;
        END IF;

        SELECT EXISTS (
            SELECT 1 FROM information_schema.columns
            WHERE table_schema = 'inscada' AND table_name = 'alarm' AND column_name = 'on_time_variable_id'
        ) INTO column_exists;

        IF column_exists AND NEW.on_time_variable_id IS NOT NULL AND NOT inscada.check_variable_exists(NEW.on_time_variable_id) THEN
            RAISE EXCEPTION 'Variable with id % does not exist', NEW.on_time_variable_id;
        END IF;
    END IF;

    -- For digital_alarm with variable_a_id
    IF TG_TABLE_NAME = 'digital_alarm' THEN
        SELECT EXISTS (
            SELECT 1 FROM information_schema.columns
            WHERE table_schema = 'inscada' AND table_name = 'digital_alarm' AND column_name = 'variable_a_id'
        ) INTO column_exists;

        IF column_exists AND NEW.variable_a_id IS NOT NULL AND NOT inscada.check_variable_exists(NEW.variable_a_id) THEN
            RAISE EXCEPTION 'Variable with id % does not exist', NEW.variable_a_id;
        END IF;

        SELECT EXISTS (
            SELECT 1 FROM information_schema.columns
            WHERE table_schema = 'inscada' AND table_name = 'digital_alarm' AND column_name = 'variable_b_id'
        ) INTO column_exists;

        IF column_exists AND NEW.variable_b_id IS NOT NULL AND NOT inscada.check_variable_exists(NEW.variable_b_id) THEN
            RAISE EXCEPTION 'Variable with id % does not exist', NEW.variable_b_id;
        END IF;
    END IF;

    -- For analog_alarm with variable_id
    IF TG_TABLE_NAME = 'analog_alarm' THEN
        SELECT EXISTS (
            SELECT 1 FROM information_schema.columns
            WHERE table_schema = 'inscada' AND table_name = 'analog_alarm' AND column_name = 'variable_id'
        ) INTO column_exists;

        IF column_exists AND NEW.variable_id IS NOT NULL AND NOT inscada.check_variable_exists(NEW.variable_id) THEN
            RAISE EXCEPTION 'Variable with id % does not exist', NEW.variable_id;
        END IF;
    END IF;

    -- For data_transfer_detail with source_var_id
    IF TG_TABLE_NAME = 'data_transfer_detail' THEN
        SELECT EXISTS (
            SELECT 1 FROM information_schema.columns
            WHERE table_schema = 'inscada' AND table_name = 'data_transfer_detail' AND column_name = 'source_var_id'
        ) INTO column_exists;

        IF column_exists AND NEW.source_var_id IS NOT NULL AND NOT inscada.check_variable_exists(NEW.source_var_id) THEN
            RAISE EXCEPTION 'Variable with id % does not exist', NEW.source_var_id;
        END IF;

        SELECT EXISTS (
            SELECT 1 FROM information_schema.columns
            WHERE table_schema = 'inscada' AND table_name = 'data_transfer_detail' AND column_name = 'target_var_id'
        ) INTO column_exists;

        IF column_exists AND NEW.target_var_id IS NOT NULL AND NOT inscada.check_variable_exists(NEW.target_var_id) THEN
            RAISE EXCEPTION 'Variable with id % does not exist', NEW.target_var_id;
        END IF;
    END IF;

    -- For report_variable with deviation_variable_id
    IF TG_TABLE_NAME = 'report_variable' THEN
        SELECT EXISTS (
            SELECT 1 FROM information_schema.columns
            WHERE table_schema = 'inscada' AND table_name = 'report_variable' AND column_name = 'deviation_variable_id'
        ) INTO column_exists;

        IF column_exists AND NEW.deviation_variable_id IS NOT NULL AND NOT inscada.check_variable_exists(NEW.deviation_variable_id) THEN
            RAISE EXCEPTION 'Variable with id % does not exist', NEW.deviation_variable_id;
        END IF;

        SELECT EXISTS (
            SELECT 1 FROM information_schema.columns
            WHERE table_schema = 'inscada' AND table_name = 'report_variable' AND column_name = 'total_variable_id'
        ) INTO column_exists;

        IF column_exists AND NEW.total_variable_id IS NOT NULL AND NOT inscada.check_variable_exists(NEW.total_variable_id) THEN
            RAISE EXCEPTION 'Variable with id % does not exist', NEW.total_variable_id;
        END IF;

        SELECT EXISTS (
            SELECT 1 FROM information_schema.columns
            WHERE table_schema = 'inscada' AND table_name = 'report_variable' AND column_name = 'variable_id'
        ) INTO column_exists;

        IF column_exists AND NEW.variable_id IS NOT NULL AND NOT inscada.check_variable_exists(NEW.variable_id) THEN
            RAISE EXCEPTION 'Variable with id % does not exist', NEW.variable_id;
        END IF;
    END IF;

    -- For trend_tag with variable_id
    IF TG_TABLE_NAME = 'trend_tag' THEN
        SELECT EXISTS (
            SELECT 1 FROM information_schema.columns
            WHERE table_schema = 'inscada' AND table_name = 'trend_tag' AND column_name = 'variable_id'
        ) INTO column_exists;

        IF column_exists AND NEW.variable_id IS NOT NULL AND NOT inscada.check_variable_exists(NEW.variable_id) THEN
            RAISE EXCEPTION 'Variable with id % does not exist', NEW.variable_id;
        END IF;
    END IF;

    -- For map_variable with variable_id
    IF TG_TABLE_NAME = 'map_variable' THEN
        SELECT EXISTS (
            SELECT 1 FROM information_schema.columns
            WHERE table_schema = 'inscada' AND table_name = 'map_variable' AND column_name = 'variable_id'
        ) INTO column_exists;

        IF column_exists AND NEW.variable_id IS NOT NULL AND NOT inscada.check_variable_exists(NEW.variable_id) THEN
            RAISE EXCEPTION 'Variable with id % does not exist', NEW.variable_id;
        END IF;
    END IF;

    -- For monitor_variable with variable_id
    IF TG_TABLE_NAME = 'monitor_variable' THEN
        SELECT EXISTS (
            SELECT 1 FROM information_schema.columns
            WHERE table_schema = 'inscada' AND table_name = 'monitor_variable' AND column_name = 'variable_id'
        ) INTO column_exists;

        IF column_exists AND NEW.variable_id IS NOT NULL AND NOT inscada.check_variable_exists(NEW.variable_id) THEN
            RAISE EXCEPTION 'Variable with id % does not exist', NEW.variable_id;
        END IF;
    END IF;

    RETURN NEW;
END;
$$ LANGUAGE plpgsql;

-- Trigger for alarm
CREATE TRIGGER tr_alarm_variable_check
    BEFORE INSERT OR UPDATE ON alarm
    FOR EACH ROW
    EXECUTE FUNCTION validate_variable_reference();

-- Trigger for analog_alarm
CREATE TRIGGER tr_analog_alarm_variable_check
    BEFORE INSERT OR UPDATE ON analog_alarm
    FOR EACH ROW
    EXECUTE FUNCTION validate_variable_reference();

-- Trigger for digital_alarm
CREATE TRIGGER tr_digital_alarm_variable_check
    BEFORE INSERT OR UPDATE ON digital_alarm
    FOR EACH ROW
    EXECUTE FUNCTION validate_variable_reference();

-- Trigger for report_variable
CREATE TRIGGER tr_report_variable_variable_check
    BEFORE INSERT OR UPDATE ON report_variable
    FOR EACH ROW
    EXECUTE FUNCTION validate_variable_reference();

-- Trigger for trend_tag
CREATE TRIGGER tr_trend_tag_variable_check
    BEFORE INSERT OR UPDATE ON trend_tag
    FOR EACH ROW
    EXECUTE FUNCTION validate_variable_reference();

-- Trigger for map_variable
CREATE TRIGGER tr_map_variable_variable_check
    BEFORE INSERT OR UPDATE ON map_variable
    FOR EACH ROW
    EXECUTE FUNCTION validate_variable_reference();

-- Trigger for data_transfer_detail
CREATE TRIGGER tr_data_transfer_detail_variable_check
    BEFORE INSERT OR UPDATE ON data_transfer_detail
    FOR EACH ROW
    EXECUTE FUNCTION validate_variable_reference();

-- Trigger for monitor_variable
CREATE TRIGGER tr_monitor_variable_variable_check
    BEFORE INSERT OR UPDATE ON monitor_variable
    FOR EACH ROW
    EXECUTE FUNCTION validate_variable_reference();


-- =============================================================================
-- PREVENT DELETION OF REFERENCED VARIABLES
-- =============================================================================

-- Function to check if a variable is referenced by any table
CREATE OR REPLACE FUNCTION is_variable_referenced(p_variable_id VARCHAR(36))
RETURNS BOOLEAN AS $$
BEGIN
    -- Return TRUE if the variable is referenced by any table
    -- Check alarm table - off_time_variable_id
    IF EXISTS (
        SELECT 1 FROM information_schema.columns
        WHERE table_schema = 'inscada' AND table_name = 'alarm' AND column_name = 'off_time_variable_id'
    ) AND EXISTS (
        SELECT 1 FROM inscada.alarm WHERE off_time_variable_id = p_variable_id
    ) THEN
        RETURN TRUE;
    END IF;

    -- Check alarm table - on_time_variable_id
    IF EXISTS (
        SELECT 1 FROM information_schema.columns
        WHERE table_schema = 'inscada' AND table_name = 'alarm' AND column_name = 'on_time_variable_id'
    ) AND EXISTS (
        SELECT 1 FROM inscada.alarm WHERE on_time_variable_id = p_variable_id
    ) THEN
        RETURN TRUE;
    END IF;

    -- Check digital_alarm - variable_a_id
    IF EXISTS (
        SELECT 1 FROM information_schema.columns
        WHERE table_schema = 'inscada' AND table_name = 'digital_alarm' AND column_name = 'variable_a_id'
    ) AND EXISTS (
        SELECT 1 FROM inscada.digital_alarm WHERE variable_a_id = p_variable_id
    ) THEN
        RETURN TRUE;
    END IF;

    -- Check digital_alarm - variable_b_id
    IF EXISTS (
        SELECT 1 FROM information_schema.columns
        WHERE table_schema = 'inscada' AND table_name = 'digital_alarm' AND column_name = 'variable_b_id'
    ) AND EXISTS (
        SELECT 1 FROM inscada.digital_alarm WHERE variable_b_id = p_variable_id
    ) THEN
        RETURN TRUE;
    END IF;

    -- Check analog_alarm - variable_id
    IF EXISTS (
        SELECT 1 FROM information_schema.columns
        WHERE table_schema = 'inscada' AND table_name = 'analog_alarm' AND column_name = 'variable_id'
    ) AND EXISTS (
        SELECT 1 FROM inscada.analog_alarm WHERE variable_id = p_variable_id
    ) THEN
        RETURN TRUE;
    END IF;

    -- Check data_transfer_detail - source_var_id
    IF EXISTS (
        SELECT 1 FROM information_schema.columns
        WHERE table_schema = 'inscada' AND table_name = 'data_transfer_detail' AND column_name = 'source_var_id'
    ) AND EXISTS (
        SELECT 1 FROM inscada.data_transfer_detail WHERE source_var_id = p_variable_id
    ) THEN
        RETURN TRUE;
    END IF;

    -- Check data_transfer_detail - target_var_id
    IF EXISTS (
        SELECT 1 FROM information_schema.columns
        WHERE table_schema = 'inscada' AND table_name = 'data_transfer_detail' AND column_name = 'target_var_id'
    ) AND EXISTS (
        SELECT 1 FROM inscada.data_transfer_detail WHERE target_var_id = p_variable_id
    ) THEN
        RETURN TRUE;
    END IF;

    -- Check report_variable - deviation_variable_id
    IF EXISTS (
        SELECT 1 FROM information_schema.columns
        WHERE table_schema = 'inscada' AND table_name = 'report_variable' AND column_name = 'deviation_variable_id'
    ) AND EXISTS (
        SELECT 1 FROM inscada.report_variable WHERE deviation_variable_id = p_variable_id
    ) THEN
        RETURN TRUE;
    END IF;

    -- Check report_variable - total_variable_id
    IF EXISTS (
        SELECT 1 FROM information_schema.columns
        WHERE table_schema = 'inscada' AND table_name = 'report_variable' AND column_name = 'total_variable_id'
    ) AND EXISTS (
        SELECT 1 FROM inscada.report_variable WHERE total_variable_id = p_variable_id
    ) THEN
        RETURN TRUE;
    END IF;

    -- Check report_variable - variable_id
    IF EXISTS (
        SELECT 1 FROM information_schema.columns
        WHERE table_schema = 'inscada' AND table_name = 'report_variable' AND column_name = 'variable_id'
    ) AND EXISTS (
        SELECT 1 FROM inscada.report_variable WHERE variable_id = p_variable_id
    ) THEN
        RETURN TRUE;
    END IF;

    -- Check trend_tag - variable_id
    IF EXISTS (
        SELECT 1 FROM information_schema.columns
        WHERE table_schema = 'inscada' AND table_name = 'trend_tag' AND column_name = 'variable_id'
    ) AND EXISTS (
        SELECT 1 FROM inscada.trend_tag WHERE variable_id = p_variable_id
    ) THEN
        RETURN TRUE;
    END IF;

    -- Check map_variable - variable_id
    IF EXISTS (
        SELECT 1 FROM information_schema.columns
        WHERE table_schema = 'inscada' AND table_name = 'map_variable' AND column_name = 'variable_id'
    ) AND EXISTS (
        SELECT 1 FROM inscada.map_variable WHERE variable_id = p_variable_id
    ) THEN
        RETURN TRUE;
    END IF;

    -- Check monitor_variable - variable_id
    IF EXISTS (
        SELECT 1 FROM information_schema.columns
        WHERE table_schema = 'inscada' AND table_name = 'monitor_variable' AND column_name = 'variable_id'
    ) AND EXISTS (
        SELECT 1 FROM inscada.monitor_variable WHERE variable_id = p_variable_id
    ) THEN
        RETURN TRUE;
    END IF;

    -- Not referenced in any table
    RETURN FALSE;
END;
$$ LANGUAGE plpgsql;

-- Function to prevent deletion of referenced variables
CREATE OR REPLACE FUNCTION prevent_referenced_variable_deletion()
RETURNS TRIGGER AS $$
DECLARE
    table_name VARCHAR;
    protocol VARCHAR;
BEGIN
    -- Get the protocol name for better error messages
    table_name := TG_TABLE_NAME;

    -- Set protocol based on table name
    CASE table_name
        WHEN 'dnp3_variable' THEN protocol := 'DNP3';
        WHEN 'modbus_variable' THEN protocol := 'MODBUS';
        WHEN 's7_variable' THEN protocol := 'S7';
        WHEN 'iec104_variable' THEN protocol := 'IEC104';
        WHEN 'iec61850_variable' THEN protocol := 'IEC61850';
        WHEN 'mqtt_variable' THEN protocol := 'MQTT';
        WHEN 'opc_ua_variable' THEN protocol := 'OPC_UA';
        WHEN 'ethernet_ip_variable' THEN protocol := 'ETHERNET_IP';
        WHEN 'fatek_variable' THEN protocol := 'FATEK';
        WHEN 'local_variable' THEN protocol := 'LOCAL';
        WHEN 'opc_da_variable' THEN protocol := 'OPC_DA';
        ELSE protocol := 'UNKNOWN';
    END CASE;

    -- Check if the variable is referenced by any table
    IF inscada.is_variable_referenced(OLD.variable_id) THEN
        RAISE EXCEPTION 'Cannot delete % variable with ID % - it is referenced by other entities. Remove all references before deleting.',
            protocol, OLD.variable_id;
    END IF;

    -- If not referenced, allow the deletion
    RETURN OLD;
END;
$$ LANGUAGE plpgsql;

-- Create DELETE triggers for each protocol variable table
CREATE TRIGGER tr_dnp3_variable_prevent_delete
    BEFORE DELETE ON dnp3_variable
    FOR EACH ROW
    EXECUTE FUNCTION prevent_referenced_variable_deletion();

CREATE TRIGGER tr_modbus_variable_prevent_delete
    BEFORE DELETE ON modbus_variable
    FOR EACH ROW
    EXECUTE FUNCTION prevent_referenced_variable_deletion();

CREATE TRIGGER tr_s7_variable_prevent_delete
    BEFORE DELETE ON s7_variable
    FOR EACH ROW
    EXECUTE FUNCTION prevent_referenced_variable_deletion();

CREATE TRIGGER tr_iec104_variable_prevent_delete
    BEFORE DELETE ON iec104_variable
    FOR EACH ROW
    EXECUTE FUNCTION prevent_referenced_variable_deletion();

CREATE TRIGGER tr_iec61850_variable_prevent_delete
    BEFORE DELETE ON iec61850_variable
    FOR EACH ROW
    EXECUTE FUNCTION prevent_referenced_variable_deletion();

CREATE TRIGGER tr_mqtt_variable_prevent_delete
    BEFORE DELETE ON mqtt_variable
    FOR EACH ROW
    EXECUTE FUNCTION prevent_referenced_variable_deletion();

CREATE TRIGGER tr_opc_ua_variable_prevent_delete
    BEFORE DELETE ON opc_ua_variable
    FOR EACH ROW
    EXECUTE FUNCTION prevent_referenced_variable_deletion();

CREATE TRIGGER tr_ethernet_ip_variable_prevent_delete
    BEFORE DELETE ON ethernet_ip_variable
    FOR EACH ROW
    EXECUTE FUNCTION prevent_referenced_variable_deletion();

CREATE TRIGGER tr_fatek_variable_prevent_delete
    BEFORE DELETE ON fatek_variable
    FOR EACH ROW
    EXECUTE FUNCTION prevent_referenced_variable_deletion();

CREATE TRIGGER tr_local_variable_prevent_delete
    BEFORE DELETE ON local_variable
    FOR EACH ROW
    EXECUTE FUNCTION prevent_referenced_variable_deletion();

CREATE TRIGGER tr_opc_da_variable_prevent_delete
    BEFORE DELETE ON opc_da_variable
    FOR EACH ROW
    EXECUTE FUNCTION prevent_referenced_variable_deletion();

-- =============================================================================
-- VARIABLE NAME UNIQUENESS
-- =============================================================================
-- Create a registry table to enforce variable name uniqueness across all protocols within a project
CREATE TABLE variable_registry (
    variable_id VARCHAR(36) NOT NULL PRIMARY KEY,
    project_id VARCHAR(36) NOT NULL, -- All variables in regular tables have project_id
    name VARCHAR(100) NOT NULL,
    protocol VARCHAR(20) NOT NULL,
    -- Unique constraint for variable names within project
    CONSTRAINT uk_variable_registry_project_name UNIQUE (project_id, name) DEFERRABLE INITIALLY DEFERRED
);

-- Create index for performance
CREATE INDEX idx_variable_registry_project ON variable_registry(project_id);

-- Populate variable_registry with existing variables from all protocol tables
-- All regular variable tables have project_id NOT NULL
INSERT INTO variable_registry (variable_id, project_id, name, protocol)
SELECT v.variable_id, v.project_id, v.name, 'DNP3'
FROM dnp3_variable v
UNION ALL
SELECT v.variable_id, v.project_id, v.name, 'MODBUS'
FROM modbus_variable v
UNION ALL
SELECT v.variable_id, v.project_id, v.name, 'S7'
FROM s7_variable v
UNION ALL
SELECT v.variable_id, v.project_id, v.name, 'IEC104'
FROM iec104_variable v
UNION ALL
SELECT v.variable_id, v.project_id, v.name, 'IEC61850'
FROM iec61850_variable v
UNION ALL
SELECT v.variable_id, v.project_id, v.name, 'MQTT'
FROM mqtt_variable v
UNION ALL
SELECT v.variable_id, v.project_id, v.name, 'OPC_UA'
FROM opc_ua_variable v
UNION ALL
SELECT v.variable_id, v.project_id, v.name, 'ETHERNET_IP'
FROM ethernet_ip_variable v
UNION ALL
SELECT v.variable_id, v.project_id, v.name, 'FATEK'
FROM fatek_variable v
UNION ALL
SELECT v.variable_id, v.project_id, v.name, 'LOCAL'
FROM local_variable v
UNION ALL
SELECT v.variable_id, v.project_id, v.name, 'OPC_DA'
FROM opc_da_variable v;

-- Function to validate variable name uniqueness within a project
CREATE OR REPLACE FUNCTION validate_variable_name_uniqueness()
RETURNS TRIGGER AS $$
BEGIN
    -- Check if variable name already exists in the same project (excluding the current variable for updates)
    IF (TG_OP = 'INSERT' OR (TG_OP = 'UPDATE' AND OLD.name != NEW.name)) THEN
        IF EXISTS (
            SELECT 1 FROM inscada.variable_registry
            WHERE project_id = NEW.project_id
            AND name = NEW.name
            AND variable_id != NEW.variable_id
        ) THEN
            RAISE EXCEPTION 'Variable name "%" already exists in project "%"', NEW.name, NEW.project_id;
        END IF;
    END IF;

    -- For INSERT operations, add to variable_registry
    IF TG_OP = 'INSERT' THEN
        INSERT INTO inscada.variable_registry (variable_id, project_id, name, protocol)
        VALUES (NEW.variable_id, NEW.project_id, NEW.name,
            CASE TG_TABLE_NAME
                WHEN 'dnp3_variable' THEN 'DNP3'
                WHEN 'modbus_variable' THEN 'MODBUS'
                WHEN 's7_variable' THEN 'S7'
                WHEN 'iec104_variable' THEN 'IEC104'
                WHEN 'iec61850_variable' THEN 'IEC61850'
                WHEN 'mqtt_variable' THEN 'MQTT'
                WHEN 'opc_ua_variable' THEN 'OPC_UA'
                WHEN 'ethernet_ip_variable' THEN 'ETHERNET_IP'
                WHEN 'fatek_variable' THEN 'FATEK'
                WHEN 'local_variable' THEN 'LOCAL'
                WHEN 'opc_da_variable' THEN 'OPC_DA'
            END
        );
    END IF;

    -- For UPDATE operations, update variable_registry
    IF TG_OP = 'UPDATE' THEN
        UPDATE inscada.variable_registry
        SET project_id = NEW.project_id, name = NEW.name
        WHERE variable_id = NEW.variable_id;
    END IF;

    RETURN NEW;
END;
$$ LANGUAGE plpgsql;

-- Function to handle variable deletion from registry
CREATE OR REPLACE FUNCTION cleanup_variable_registry()
RETURNS TRIGGER AS $$
BEGIN
    -- Remove from variable_registry when variable is deleted
    DELETE FROM inscada.variable_registry WHERE variable_id = OLD.variable_id;
    RETURN OLD;
END;
$$ LANGUAGE plpgsql;

-- DNP3 Variable triggers
CREATE TRIGGER tr_dnp3_variable_name_uniqueness
    BEFORE INSERT OR UPDATE ON dnp3_variable
    FOR EACH ROW
    EXECUTE FUNCTION validate_variable_name_uniqueness();

CREATE TRIGGER tr_dnp3_variable_cleanup
    AFTER DELETE ON dnp3_variable
    FOR EACH ROW
    EXECUTE FUNCTION cleanup_variable_registry();

-- Modbus Variable triggers
CREATE TRIGGER tr_modbus_variable_name_uniqueness
    BEFORE INSERT OR UPDATE ON modbus_variable
    FOR EACH ROW
    EXECUTE FUNCTION validate_variable_name_uniqueness();

CREATE TRIGGER tr_modbus_variable_cleanup
    AFTER DELETE ON modbus_variable
    FOR EACH ROW
    EXECUTE FUNCTION cleanup_variable_registry();

-- S7 Variable triggers
CREATE TRIGGER tr_s7_variable_name_uniqueness
    BEFORE INSERT OR UPDATE ON s7_variable
    FOR EACH ROW
    EXECUTE FUNCTION validate_variable_name_uniqueness();

CREATE TRIGGER tr_s7_variable_cleanup
    AFTER DELETE ON s7_variable
    FOR EACH ROW
    EXECUTE FUNCTION cleanup_variable_registry();

-- IEC104 Variable triggers
CREATE TRIGGER tr_iec104_variable_name_uniqueness
    BEFORE INSERT OR UPDATE ON iec104_variable
    FOR EACH ROW
    EXECUTE FUNCTION validate_variable_name_uniqueness();

CREATE TRIGGER tr_iec104_variable_cleanup
    AFTER DELETE ON iec104_variable
    FOR EACH ROW
    EXECUTE FUNCTION cleanup_variable_registry();

-- IEC61850 Variable triggers
CREATE TRIGGER tr_iec61850_variable_name_uniqueness
    BEFORE INSERT OR UPDATE ON iec61850_variable
    FOR EACH ROW
    EXECUTE FUNCTION validate_variable_name_uniqueness();

CREATE TRIGGER tr_iec61850_variable_cleanup
    AFTER DELETE ON iec61850_variable
    FOR EACH ROW
    EXECUTE FUNCTION cleanup_variable_registry();

-- MQTT Variable triggers
CREATE TRIGGER tr_mqtt_variable_name_uniqueness
    BEFORE INSERT OR UPDATE ON mqtt_variable
    FOR EACH ROW
    EXECUTE FUNCTION validate_variable_name_uniqueness();

CREATE TRIGGER tr_mqtt_variable_cleanup
    AFTER DELETE ON mqtt_variable
    FOR EACH ROW
    EXECUTE FUNCTION cleanup_variable_registry();

-- OPC_UA Variable triggers
CREATE TRIGGER tr_opc_ua_variable_name_uniqueness
    BEFORE INSERT OR UPDATE ON opc_ua_variable
    FOR EACH ROW
    EXECUTE FUNCTION validate_variable_name_uniqueness();

CREATE TRIGGER tr_opc_ua_variable_cleanup
    AFTER DELETE ON opc_ua_variable
    FOR EACH ROW
    EXECUTE FUNCTION cleanup_variable_registry();

-- Ethernet_IP Variable triggers
CREATE TRIGGER tr_ethernet_ip_variable_name_uniqueness
    BEFORE INSERT OR UPDATE ON ethernet_ip_variable
    FOR EACH ROW
    EXECUTE FUNCTION validate_variable_name_uniqueness();

CREATE TRIGGER tr_ethernet_ip_variable_cleanup
    AFTER DELETE ON ethernet_ip_variable
    FOR EACH ROW
    EXECUTE FUNCTION cleanup_variable_registry();

-- Fatek Variable triggers
CREATE TRIGGER tr_fatek_variable_name_uniqueness
    BEFORE INSERT OR UPDATE ON fatek_variable
    FOR EACH ROW
    EXECUTE FUNCTION validate_variable_name_uniqueness();

CREATE TRIGGER tr_fatek_variable_cleanup
    AFTER DELETE ON fatek_variable
    FOR EACH ROW
    EXECUTE FUNCTION cleanup_variable_registry();

-- Local Variable triggers
CREATE TRIGGER tr_local_variable_name_uniqueness
    BEFORE INSERT OR UPDATE ON local_variable
    FOR EACH ROW
    EXECUTE FUNCTION validate_variable_name_uniqueness();

CREATE TRIGGER tr_local_variable_cleanup
    AFTER DELETE ON local_variable
    FOR EACH ROW
    EXECUTE FUNCTION cleanup_variable_registry();

-- OPC_DA Variable triggers
CREATE TRIGGER tr_opc_da_variable_name_uniqueness
    BEFORE INSERT OR UPDATE ON opc_da_variable
    FOR EACH ROW
    EXECUTE FUNCTION validate_variable_name_uniqueness();

CREATE TRIGGER tr_opc_da_variable_cleanup
    AFTER DELETE ON opc_da_variable
    FOR EACH ROW
    EXECUTE FUNCTION cleanup_variable_registry();