-- Step 1: Find and update duplicate emails.
-- This script identifies users with duplicate emails and appends a unique suffix
-- to all but the oldest entry for each email address.
-- For example, 'test@example.com' for a second user becomes 'test+dup.some-uuid@example.com'.
-- This preserves the user account while ensuring uniqueness for the constraint.
WITH duplicates AS (
    SELECT
        user_id,
        ROW_NUMBER() OVER(PARTITION BY LOWER(email) ORDER BY insert_dttm ASC) as rn
    FROM
        inscada."users"
    WHERE
        email IS NOT NULL AND email != '' AND deleted_dttm IS NULL
)
UPDATE
    inscada."users"
SET
    -- Splitting the email at '@' to insert the unique suffix correctly.
    email = split_part(email, '@', 1) || '+dup.' || duplicates.user_id || '@' || split_part(email, '@', 2)
FROM
    duplicates
WHERE
    inscada."users".user_id = duplicates.user_id AND duplicates.rn > 1;

-- Step 2: Add the unique constraint to the email column.
-- This will only succeed if Step 1 has successfully resolved all duplicates.
-- We use a partial index to enforce uniqueness only on active (not soft-deleted) users.
CREATE UNIQUE INDEX IF NOT EXISTS uk_user_email_lower ON inscada."users" (LOWER(email)) WHERE deleted_dttm IS NULL;