-- This migration fixes data inconsistencies from previous migrations (V233, V234, V281) that used
-- non-deterministic UUIDs for space-specific entities and join tables.
-- It recalculates all affected UUIDs using a deterministic method (MD5 hash) that requires
-- no extensions, ensuring all nodes in a cluster converge on the same IDs.

DO $$
BEGIN
    -- === Step 1: Create temporary mapping tables for all affected entities ===
    -- Temporarily drop foreign key constraints to allow out-of-order PK/FK updates.
    -- They will be re-added at the end of the script.
    ALTER TABLE role_permissions DROP CONSTRAINT IF EXISTS role_permissions_roles_role_id_fk;
    ALTER TABLE role_permissions DROP CONSTRAINT IF EXISTS role_permissions_permissions_permission_id_fk;
    ALTER TABLE role_menus DROP CONSTRAINT IF EXISTS role_menus_roles_role_id_fk;
    ALTER TABLE role_menus DROP CONSTRAINT IF EXISTS role_menus_menus_menu_id_fk;
    ALTER TABLE role_members DROP CONSTRAINT IF EXISTS role_members_roles_role_id_fk;
    ALTER TABLE custom_menu DROP CONSTRAINT IF EXISTS fk_custom_menu_menu;
    ALTER TABLE custom_menu DROP CONSTRAINT IF EXISTS fk_custom_menu_parent;

    -- Main entities
    CREATE TEMP TABLE permission_id_map (old_id TEXT PRIMARY KEY, new_id TEXT NOT NULL);
    CREATE TEMP TABLE role_id_map (old_id TEXT PRIMARY KEY, new_id TEXT NOT NULL);
    CREATE TEMP TABLE menu_id_map (old_id TEXT PRIMARY KEY, new_id TEXT NOT NULL); -- Menus used TEXT for IDs in a previous migration
    -- Join table entities
    CREATE TEMP TABLE role_members_id_map (old_id TEXT PRIMARY KEY, new_id TEXT NOT NULL);
    CREATE TEMP TABLE space_users_id_map (old_id TEXT PRIMARY KEY, new_id TEXT NOT NULL);
    CREATE TEMP TABLE role_permissions_id_map (old_id TEXT PRIMARY KEY, new_id TEXT NOT NULL);
    CREATE TEMP TABLE role_menus_id_map (old_id TEXT PRIMARY KEY, new_id TEXT NOT NULL);

    -- === Step 2: Populate mapping tables for the MAIN entities (permissions, roles, menus) ===
    -- The new UUID is based on the entity's name and its space_id.
    INSERT INTO permission_id_map (old_id, new_id)
    SELECT permission_id, uuid(md5(name || '::' || space_id::text))::text
    FROM permissions WHERE space_id IS NOT NULL;

    INSERT INTO role_id_map (old_id, new_id)
    SELECT role_id, uuid(md5(name || '::' || space_id::text))::text
    FROM roles WHERE space_id IS NOT NULL;

    -- Includes both standard and custom menus
    INSERT INTO menu_id_map (old_id, new_id)
    SELECT menu_id, uuid(md5(name || '::' || space_id::text))::text
    FROM menus WHERE space_id IS NOT NULL;

    -- === Step 3: Update FOREIGN KEY references in join tables BEFORE updating primary keys ===
    -- This is the most critical step to avoid breaking constraints.
    UPDATE role_permissions SET role_id = role_id_map.new_id::uuid FROM role_id_map WHERE role_permissions.role_id::text = role_id_map.old_id;
    UPDATE role_permissions SET permission_id = permission_id_map.new_id::uuid FROM permission_id_map WHERE role_permissions.permission_id::text = permission_id_map.old_id;

    UPDATE role_menus SET role_id = role_id_map.new_id::uuid FROM role_id_map WHERE role_menus.role_id::text = role_id_map.old_id;
    UPDATE role_menus SET menu_id = menu_id_map.new_id FROM menu_id_map WHERE role_menus.menu_id = menu_id_map.old_id;

    UPDATE role_members SET role_id = role_id_map.new_id::uuid FROM role_id_map WHERE role_members.role_id::text = role_id_map.old_id;

    -- Update the self-referencing FK in custom_menu
    UPDATE custom_menu SET parent_menu_id = menu_id_map.new_id FROM menu_id_map WHERE custom_menu.parent_menu_id = menu_id_map.old_id;

    -- === Step 4: Update the PRIMARY KEYs in the MAIN tables and inheritance children ===
    -- Update child table (custom_menu) before parent (menus)
    UPDATE custom_menu SET menu_id = menu_id_map.new_id FROM menu_id_map WHERE custom_menu.menu_id = menu_id_map.old_id;

    UPDATE permissions SET permission_id = permission_id_map.new_id::uuid FROM permission_id_map WHERE permissions.permission_id::text = permission_id_map.old_id;
    UPDATE roles SET role_id = role_id_map.new_id::uuid FROM role_id_map WHERE roles.role_id::text = role_id_map.old_id;
    UPDATE menus SET menu_id = menu_id_map.new_id FROM menu_id_map WHERE menus.menu_id = menu_id_map.old_id;

    -- === Step 5: Now that FKs are correct, populate mapping tables for the JOIN tables ===
    -- The new UUID is based on the (now deterministic) foreign keys.
    -- This step must happen AFTER the main entity FKs have been updated in Step 3.
    INSERT INTO role_permissions_id_map (old_id, new_id)
    SELECT
        role_permission_id,
        uuid(md5(role_id::text || '::' || permission_id::text))::text
    FROM role_permissions;

    INSERT INTO space_users_id_map (old_id, new_id)
    SELECT
        space_user_id,
        uuid(md5(user_id::text || '::' || space_id::text))::text
    FROM space_users;

    INSERT INTO role_members_id_map (old_id, new_id)
    SELECT
        role_member_id,
        uuid(md5(user_id::text || '::' || role_id::text))::text
    FROM role_members;

    INSERT INTO role_menus_id_map (old_id, new_id)
    SELECT
        role_menu_id,
        uuid(md5(role_id::text || '::' || menu_id::text))::text
    FROM role_menus;

    -- === Step 6: Update the PRIMARY KEYs in the JOIN tables to their new deterministic IDs ===
    UPDATE role_permissions
    SET role_permission_id = role_permissions_id_map.new_id::uuid
    FROM role_permissions_id_map
    WHERE role_permissions.role_permission_id::text = role_permissions_id_map.old_id;

    UPDATE space_users
    SET space_user_id = space_users_id_map.new_id::uuid
    FROM space_users_id_map
    WHERE space_users.space_user_id::text = space_users_id_map.old_id;

    UPDATE role_members
    SET role_member_id = role_members_id_map.new_id::uuid
    FROM role_members_id_map
    WHERE role_members.role_member_id::text = role_members_id_map.old_id;

    UPDATE role_menus
    SET role_menu_id = role_menus_id_map.new_id::uuid
    FROM role_menus_id_map
    WHERE role_menus.role_menu_id::text = role_menus_id_map.old_id;

    -- === Step 7: Clean up all temporary tables ===
    -- Re-create the foreign key constraints to ensure data integrity.
    ALTER TABLE role_permissions ADD CONSTRAINT role_permissions_roles_role_id_fk FOREIGN KEY (role_id) REFERENCES roles(role_id);
    ALTER TABLE role_permissions ADD CONSTRAINT role_permissions_permissions_permission_id_fk FOREIGN KEY (permission_id) REFERENCES permissions(permission_id);
    ALTER TABLE role_menus ADD CONSTRAINT role_menus_roles_role_id_fk FOREIGN KEY (role_id) REFERENCES roles(role_id);
    ALTER TABLE role_menus ADD CONSTRAINT role_menus_menus_menu_id_fk FOREIGN KEY (menu_id) REFERENCES menus(menu_id);
    ALTER TABLE role_members ADD CONSTRAINT role_members_roles_role_id_fk FOREIGN KEY (role_id) REFERENCES roles(role_id);
    ALTER TABLE custom_menu ADD CONSTRAINT fk_custom_menu_menu FOREIGN KEY (menu_id) REFERENCES menus(menu_id);
    ALTER TABLE custom_menu ADD CONSTRAINT fk_custom_menu_parent FOREIGN KEY (parent_menu_id) REFERENCES menus(menu_id);


    DROP TABLE permission_id_map;
    DROP TABLE role_id_map;
    DROP TABLE menu_id_map;
    DROP TABLE role_menus_id_map;
    DROP TABLE role_permissions_id_map;
    DROP TABLE space_users_id_map;
    DROP TABLE role_members_id_map;

END $$;