-- =============================================================================
-- CONNECTION UNIQUENESS
-- =============================================================================

CREATE TABLE connection_registry (
    conn_id VARCHAR(36) NOT NULL PRIMARY KEY,
    project_id VARCHAR(36) NOT NULL,
    name VARCHAR(100) NOT NULL,
    protocol VARCHAR(20) NOT NULL,
    -- Unique constraint for connection names within project
    CONSTRAINT uk_connection_registry_project_name UNIQUE (project_id, name) DEFERRABLE INITIALLY DEFERRED
);

-- Create index for performance
CREATE INDEX idx_connection_registry_project ON connection_registry(project_id);

-- Populate connection_registry with existing connections from all protocol tables
INSERT INTO connection_registry (conn_id, project_id, name, protocol)
SELECT conn_id, project_id, name, 'DNP3' FROM dnp3_connection
UNION ALL
SELECT conn_id, project_id, name, 'MODBUS' FROM modbus_connection
UNION ALL
SELECT conn_id, project_id, name, 'S7' FROM s7_connection
UNION ALL
SELECT conn_id, project_id, name, 'IEC104' FROM iec104_connection
UNION ALL
SELECT conn_id, project_id, name, 'IEC61850' FROM iec61850_connection
UNION ALL
SELECT conn_id, project_id, name, 'MQTT' FROM mqtt_connection
UNION ALL
SELECT conn_id, project_id, name, 'OPC_UA' FROM opc_ua_connection
UNION ALL
SELECT conn_id, project_id, name, 'ETHERNET_IP' FROM ethernet_ip_connection
UNION ALL
SELECT conn_id, project_id, name, 'FATEK' FROM fatek_connection
UNION ALL
SELECT conn_id, project_id, name, 'LOCAL' FROM local_connection
UNION ALL
SELECT conn_id, project_id, name, 'OPC_DA' FROM opc_da_connection;

-- Function to validate connection name uniqueness within a project
CREATE OR REPLACE FUNCTION validate_connection_name_uniqueness()
RETURNS TRIGGER AS $$
BEGIN
    -- Check if connection name already exists in the same project (excluding the current connection for updates)
    IF (TG_OP = 'INSERT' OR (TG_OP = 'UPDATE' AND (OLD.name != NEW.name OR OLD.project_id != NEW.project_id))) THEN
        IF EXISTS (
            SELECT 1 FROM inscada.connection_registry
            WHERE project_id = NEW.project_id
            AND name = NEW.name
            AND conn_id != NEW.conn_id
        ) THEN
            RAISE EXCEPTION 'Connection name "%" already exists in project "%"', NEW.name, NEW.project_id;
        END IF;
    END IF;

    -- For INSERT operations, add to connection_registry
    IF TG_OP = 'INSERT' THEN
        INSERT INTO inscada.connection_registry (conn_id, project_id, name, protocol)
        VALUES (NEW.conn_id, NEW.project_id, NEW.name,
            CASE TG_TABLE_NAME
                WHEN 'dnp3_connection' THEN 'DNP3'
                WHEN 'modbus_connection' THEN 'MODBUS'
                WHEN 's7_connection' THEN 'S7'
                WHEN 'iec104_connection' THEN 'IEC104'
                WHEN 'iec61850_connection' THEN 'IEC61850'
                WHEN 'mqtt_connection' THEN 'MQTT'
                WHEN 'opc_ua_connection' THEN 'OPC_UA'
                WHEN 'ethernet_ip_connection' THEN 'ETHERNET_IP'
                WHEN 'fatek_connection' THEN 'FATEK'
                WHEN 'local_connection' THEN 'LOCAL'
                WHEN 'opc_da_connection' THEN 'OPC_DA'
            END
        );
    END IF;

    -- For UPDATE operations, update connection_registry
    IF TG_OP = 'UPDATE' THEN
        UPDATE inscada.connection_registry
        SET project_id = NEW.project_id, name = NEW.name
        WHERE conn_id = NEW.conn_id;
    END IF;

    RETURN NEW;
END;
$$ LANGUAGE plpgsql;

-- Function to handle connection deletion from registry
CREATE OR REPLACE FUNCTION cleanup_connection_registry()
RETURNS TRIGGER AS $$
BEGIN
    -- Remove from connection_registry when connection is deleted
    DELETE FROM inscada.connection_registry WHERE conn_id = OLD.conn_id;
    RETURN OLD;
END;
$$ LANGUAGE plpgsql;

-- Connection triggers for all 11 protocols
CREATE TRIGGER tr_dnp3_connection_name_uniqueness
    BEFORE INSERT OR UPDATE ON dnp3_connection
    FOR EACH ROW
    EXECUTE FUNCTION validate_connection_name_uniqueness();

CREATE TRIGGER tr_dnp3_connection_cleanup
    AFTER DELETE ON dnp3_connection
    FOR EACH ROW
    EXECUTE FUNCTION cleanup_connection_registry();

CREATE TRIGGER tr_modbus_connection_name_uniqueness
    BEFORE INSERT OR UPDATE ON modbus_connection
    FOR EACH ROW
    EXECUTE FUNCTION validate_connection_name_uniqueness();

CREATE TRIGGER tr_modbus_connection_cleanup
    AFTER DELETE ON modbus_connection
    FOR EACH ROW
    EXECUTE FUNCTION cleanup_connection_registry();

CREATE TRIGGER tr_s7_connection_name_uniqueness
    BEFORE INSERT OR UPDATE ON s7_connection
    FOR EACH ROW
    EXECUTE FUNCTION validate_connection_name_uniqueness();

CREATE TRIGGER tr_s7_connection_cleanup
    AFTER DELETE ON s7_connection
    FOR EACH ROW
    EXECUTE FUNCTION cleanup_connection_registry();

CREATE TRIGGER tr_iec104_connection_name_uniqueness
    BEFORE INSERT OR UPDATE ON iec104_connection
    FOR EACH ROW
    EXECUTE FUNCTION validate_connection_name_uniqueness();

CREATE TRIGGER tr_iec104_connection_cleanup
    AFTER DELETE ON iec104_connection
    FOR EACH ROW
    EXECUTE FUNCTION cleanup_connection_registry();

CREATE TRIGGER tr_iec61850_connection_name_uniqueness
    BEFORE INSERT OR UPDATE ON iec61850_connection
    FOR EACH ROW
    EXECUTE FUNCTION validate_connection_name_uniqueness();

CREATE TRIGGER tr_iec61850_connection_cleanup
    AFTER DELETE ON iec61850_connection
    FOR EACH ROW
    EXECUTE FUNCTION cleanup_connection_registry();

CREATE TRIGGER tr_mqtt_connection_name_uniqueness
    BEFORE INSERT OR UPDATE ON mqtt_connection
    FOR EACH ROW
    EXECUTE FUNCTION validate_connection_name_uniqueness();

CREATE TRIGGER tr_mqtt_connection_cleanup
    AFTER DELETE ON mqtt_connection
    FOR EACH ROW
    EXECUTE FUNCTION cleanup_connection_registry();

CREATE TRIGGER tr_opc_ua_connection_name_uniqueness
    BEFORE INSERT OR UPDATE ON opc_ua_connection
    FOR EACH ROW
    EXECUTE FUNCTION validate_connection_name_uniqueness();

CREATE TRIGGER tr_opc_ua_connection_cleanup
    AFTER DELETE ON opc_ua_connection
    FOR EACH ROW
    EXECUTE FUNCTION cleanup_connection_registry();

CREATE TRIGGER tr_ethernet_ip_connection_name_uniqueness
    BEFORE INSERT OR UPDATE ON ethernet_ip_connection
    FOR EACH ROW
    EXECUTE FUNCTION validate_connection_name_uniqueness();

CREATE TRIGGER tr_ethernet_ip_connection_cleanup
    AFTER DELETE ON ethernet_ip_connection
    FOR EACH ROW
    EXECUTE FUNCTION cleanup_connection_registry();

CREATE TRIGGER tr_fatek_connection_name_uniqueness
    BEFORE INSERT OR UPDATE ON fatek_connection
    FOR EACH ROW
    EXECUTE FUNCTION validate_connection_name_uniqueness();

CREATE TRIGGER tr_fatek_connection_cleanup
    AFTER DELETE ON fatek_connection
    FOR EACH ROW
    EXECUTE FUNCTION cleanup_connection_registry();

CREATE TRIGGER tr_local_connection_name_uniqueness
    BEFORE INSERT OR UPDATE ON local_connection
    FOR EACH ROW
    EXECUTE FUNCTION validate_connection_name_uniqueness();

CREATE TRIGGER tr_local_connection_cleanup
    AFTER DELETE ON local_connection
    FOR EACH ROW
    EXECUTE FUNCTION cleanup_connection_registry();

CREATE TRIGGER tr_opc_da_connection_name_uniqueness
    BEFORE INSERT OR UPDATE ON opc_da_connection
    FOR EACH ROW
    EXECUTE FUNCTION validate_connection_name_uniqueness();

CREATE TRIGGER tr_opc_da_connection_cleanup
    AFTER DELETE ON opc_da_connection
    FOR EACH ROW
    EXECUTE FUNCTION cleanup_connection_registry();

-- =============================================================================
-- DEVICE UNIQUENESS (simple - unique within connection)
-- =============================================================================

-- Device unique constraints (11 protocols) - unique within connection and space
ALTER TABLE dnp3_device ADD CONSTRAINT uk_dnp3_device_name UNIQUE (conn_id, name);
ALTER TABLE modbus_device ADD CONSTRAINT uk_modbus_device_name UNIQUE (conn_id, name);
ALTER TABLE s7_device ADD CONSTRAINT uk_s7_device_name UNIQUE (conn_id, name);
ALTER TABLE iec104_device ADD CONSTRAINT uk_iec104_device_name UNIQUE (conn_id, name);
ALTER TABLE iec61850_device ADD CONSTRAINT uk_iec61850_device_name UNIQUE (conn_id, name);
ALTER TABLE mqtt_device ADD CONSTRAINT uk_mqtt_device_name UNIQUE (conn_id, name);
ALTER TABLE opc_ua_device ADD CONSTRAINT uk_opc_ua_device_name UNIQUE (conn_id, name);
ALTER TABLE ethernet_ip_device ADD CONSTRAINT uk_ethernet_ip_device_name UNIQUE (conn_id, name);
ALTER TABLE fatek_device ADD CONSTRAINT uk_fatek_device_name UNIQUE (conn_id, name);
ALTER TABLE local_device ADD CONSTRAINT uk_local_device_name UNIQUE (conn_id, name);
ALTER TABLE opc_da_device ADD CONSTRAINT uk_opc_da_device_name UNIQUE (conn_id, name);

-- =============================================================================
-- FRAME UNIQUENESS (simple - unique within device)
-- =============================================================================

-- Frame unique constraints (11 protocols) - unique within device
ALTER TABLE dnp3_frame ADD CONSTRAINT uk_dnp3_frame_name UNIQUE (device_id, name);
ALTER TABLE modbus_frame ADD CONSTRAINT uk_modbus_frame_name UNIQUE (device_id, name);
ALTER TABLE s7_frame ADD CONSTRAINT uk_s7_frame_name UNIQUE (device_id, name);
ALTER TABLE iec104_frame ADD CONSTRAINT uk_iec104_frame_name UNIQUE (device_id, name);
ALTER TABLE iec61850_frame ADD CONSTRAINT uk_iec61850_frame_name UNIQUE (device_id, name);
ALTER TABLE mqtt_frame ADD CONSTRAINT uk_mqtt_frame_name UNIQUE (device_id, name);
ALTER TABLE opc_ua_frame ADD CONSTRAINT uk_opc_ua_frame_name UNIQUE (device_id, name);
ALTER TABLE ethernet_ip_frame ADD CONSTRAINT uk_ethernet_ip_frame_name UNIQUE (device_id, name);
ALTER TABLE fatek_frame ADD CONSTRAINT uk_fatek_frame_name UNIQUE (device_id, name);
ALTER TABLE local_frame ADD CONSTRAINT uk_local_frame_name UNIQUE (device_id, name);
ALTER TABLE opc_da_frame ADD CONSTRAINT uk_opc_da_frame_name UNIQUE (device_id, name);

-- =============================================================================
-- TEMPLATE TABLE UNIQUENESS CONSTRAINTS
-- =============================================================================

-- Device template unique constraints (11 protocols) - unique device template names within space
ALTER TABLE dnp3_device_template ADD CONSTRAINT uk_dnp3_device_template_name UNIQUE (space_id, name);
ALTER TABLE modbus_device_template ADD CONSTRAINT uk_modbus_device_template_name UNIQUE (space_id, name);
ALTER TABLE s7_device_template ADD CONSTRAINT uk_s7_device_template_name UNIQUE (space_id, name);
ALTER TABLE iec104_device_template ADD CONSTRAINT uk_iec104_device_template_name UNIQUE (space_id, name);
ALTER TABLE iec61850_device_template ADD CONSTRAINT uk_iec61850_device_template_name UNIQUE (space_id, name);
ALTER TABLE mqtt_device_template ADD CONSTRAINT uk_mqtt_device_template_name UNIQUE (space_id, name);
ALTER TABLE opc_ua_device_template ADD CONSTRAINT uk_opc_ua_device_template_name UNIQUE (space_id, name);
ALTER TABLE ethernet_ip_device_template ADD CONSTRAINT uk_ethernet_ip_device_template_name UNIQUE (space_id, name);
ALTER TABLE fatek_device_template ADD CONSTRAINT uk_fatek_device_template_name UNIQUE (space_id, name);
ALTER TABLE local_device_template ADD CONSTRAINT uk_local_device_template_name UNIQUE (space_id, name);
ALTER TABLE opc_da_device_template ADD CONSTRAINT uk_opc_da_device_template_name UNIQUE (space_id, name);

-- Frame template unique constraints (11 protocols) - unique within device template
ALTER TABLE dnp3_frame_template ADD CONSTRAINT uk_dnp3_frame_template_name UNIQUE (device_id, name);
ALTER TABLE modbus_frame_template ADD CONSTRAINT uk_modbus_frame_template_name UNIQUE (device_id, name);
ALTER TABLE s7_frame_template ADD CONSTRAINT uk_s7_frame_template_name UNIQUE (device_id, name);
ALTER TABLE iec104_frame_template ADD CONSTRAINT uk_iec104_frame_template_name UNIQUE (device_id, name);
ALTER TABLE iec61850_frame_template ADD CONSTRAINT uk_iec61850_frame_template_name UNIQUE (device_id, name);
ALTER TABLE mqtt_frame_template ADD CONSTRAINT uk_mqtt_frame_template_name UNIQUE (device_id, name);
ALTER TABLE opc_ua_frame_template ADD CONSTRAINT uk_opc_ua_frame_template_name UNIQUE (device_id, name);
ALTER TABLE ethernet_ip_frame_template ADD CONSTRAINT uk_ethernet_ip_frame_template_name UNIQUE (device_id, name);
ALTER TABLE fatek_frame_template ADD CONSTRAINT uk_fatek_frame_template_name UNIQUE (device_id, name);
ALTER TABLE local_frame_template ADD CONSTRAINT uk_local_frame_template_name UNIQUE (device_id, name);
ALTER TABLE opc_da_frame_template ADD CONSTRAINT uk_opc_da_frame_template_name UNIQUE (device_id, name);

-- Variable template unique constraints (11 protocols) - unique within frame template
ALTER TABLE dnp3_variable_template ADD CONSTRAINT uk_dnp3_variable_template_name UNIQUE (frame_id, name);
ALTER TABLE modbus_variable_template ADD CONSTRAINT uk_modbus_variable_template_name UNIQUE (frame_id, name);
ALTER TABLE s7_variable_template ADD CONSTRAINT uk_s7_variable_template_name UNIQUE (frame_id, name);
ALTER TABLE iec104_variable_template ADD CONSTRAINT uk_iec104_variable_template_name UNIQUE (frame_id, name);
ALTER TABLE iec61850_variable_template ADD CONSTRAINT uk_iec61850_variable_template_name UNIQUE (frame_id, name);
ALTER TABLE mqtt_variable_template ADD CONSTRAINT uk_mqtt_variable_template_name UNIQUE (frame_id, name);
ALTER TABLE opc_ua_variable_template ADD CONSTRAINT uk_opc_ua_variable_template_name UNIQUE (frame_id, name);
ALTER TABLE ethernet_ip_variable_template ADD CONSTRAINT uk_ethernet_ip_variable_template_name UNIQUE (frame_id, name);
ALTER TABLE fatek_variable_template ADD CONSTRAINT uk_fatek_variable_template_name UNIQUE (frame_id, name);
ALTER TABLE local_variable_template ADD CONSTRAINT uk_local_variable_template_name UNIQUE (frame_id, name);
ALTER TABLE opc_da_variable_template ADD CONSTRAINT uk_opc_da_variable_template_name UNIQUE (frame_id, name);