-- This migration script handles the transition to case-insensitive space names.
-- It safely resolves any existing duplicate spaces (e.g., 'Default' and 'default') by
-- renaming the duplicates with a unique suffix (e.g., 'default-a4e1').
-- It then creates a new, case-insensitive unique index that respects soft deletes.

-- Step 1: Find all spaces that are case-insensitive duplicates
-- and update their names with a unique suffix.
WITH ranked_spaces AS (
    SELECT
        space_id,
        name,
        -- Rank spaces within each duplicate group. The one with the earliest creation date gets rank 1.
        ROW_NUMBER() OVER(
            PARTITION BY LOWER(name)
            ORDER BY insert_dttm, space_id
        ) as rn
    FROM
        inscada.space
    WHERE
        deleted_dttm IS NULL
),
spaces_to_rename AS (
    -- Select all spaces that are not the first one in their duplicate group (rn > 1).
    -- These are the ones we need to rename.
    SELECT
        space_id,
        -- Append a short, unique suffix from the space's own ID for deterministic renaming.
        name || '-' || SUBSTRING(REPLACE(space_id::text, '-', ''), 1, 4) as new_name
    FROM
        ranked_spaces
    WHERE
        rn > 1
)
UPDATE
    inscada.space s
SET
    name = str.new_name
FROM
    spaces_to_rename str
WHERE
    s.space_id = str.space_id;

-- Step 2: Drop the old case-sensitive unique index by its known name.
DROP INDEX IF EXISTS inscada.space_name_uindex;

-- Step 3: Create a new, case-insensitive unique index that only applies to non-deleted rows.
CREATE UNIQUE INDEX IF NOT EXISTS space_lower_name_unique_idx
ON inscada.space (LOWER(name))
WHERE deleted_dttm IS NULL;