-- Migration script to change all primary keys from integer/serial to text in inscada schema
-- This script handles sequences, indexes, foreign keys, and primary keys

-- Step 1: Drop all sequences in the inscada schema
DO $$
DECLARE
    seq RECORD;
BEGIN
    FOR seq IN SELECT sequence_name FROM information_schema.sequences WHERE sequence_schema = 'inscada' LOOP
        EXECUTE 'DROP SEQUENCE IF EXISTS inscada.' || quote_ident(seq.sequence_name) || ' CASCADE';
    END LOOP;
END $$;

-- Step 2: Store index definitions before dropping them (but exclude constraint-backed indexes)
CREATE TEMP TABLE temp_index_definitions AS
SELECT
    i.relname as index_name,
    t.relname as table_name,
    pg_get_indexdef(i.oid) as index_definition
FROM pg_class t
JOIN pg_index ix ON t.oid = ix.indrelid
JOIN pg_class i ON i.oid = ix.indexrelid
JOIN pg_namespace n ON n.oid = t.relnamespace
WHERE n.nspname = 'inscada'
  AND i.relkind = 'i'
  AND NOT ix.indisprimary  -- Exclude primary key indexes
  AND NOT EXISTS (
    -- Exclude indexes that back constraints (unique, foreign key, etc.)
    SELECT 1 FROM pg_constraint c
    WHERE c.conindid = i.oid
  )
  AND EXISTS (
    SELECT 1 FROM pg_attribute a
    WHERE a.attrelid = t.oid
      AND a.attnum = ANY(ix.indkey)
      AND a.atttypid IN ('int4'::regtype, 'int8'::regtype)  -- Include both int4 and int8
  );

-- Step 3: Store foreign key constraints for recreation later
CREATE TEMP TABLE temp_foreign_keys AS
SELECT
    tc.constraint_name,
    tc.table_name,
    kcu.column_name,
    ccu.table_name AS foreign_table_name,
    ccu.column_name AS foreign_column_name,
    rc.delete_rule,
    rc.update_rule
FROM information_schema.table_constraints tc
JOIN information_schema.key_column_usage kcu
    ON tc.constraint_name = kcu.constraint_name
    AND tc.table_schema = kcu.table_schema
JOIN information_schema.constraint_column_usage ccu
    ON ccu.constraint_name = tc.constraint_name
    AND ccu.table_schema = tc.table_schema
JOIN information_schema.referential_constraints rc
    ON rc.constraint_name = tc.constraint_name
    AND rc.constraint_schema = tc.table_schema
WHERE tc.constraint_type = 'FOREIGN KEY'
  AND tc.table_schema = 'inscada';

-- Step 4: Drop all foreign key constraints
DO $$
DECLARE
    r RECORD;
BEGIN
    FOR r IN
        SELECT constraint_name, table_name
        FROM information_schema.table_constraints
        WHERE table_schema = 'inscada'
          AND constraint_type = 'FOREIGN KEY'
    LOOP
        EXECUTE format('ALTER TABLE inscada.%I DROP CONSTRAINT IF EXISTS %I;', r.table_name, r.constraint_name);
        RAISE NOTICE 'Dropped foreign key constraint: %.%', r.table_name, r.constraint_name;
    END LOOP;
END $$;

-- Step 5: Drop all non-primary, non-constraint-backed indexes on integer columns
DO $$
DECLARE
    r RECORD;
BEGIN
    FOR r IN
        SELECT i.relname as index_name
        FROM pg_class t
        JOIN pg_index ix ON t.oid = ix.indrelid
        JOIN pg_class i ON i.oid = ix.indexrelid
        JOIN pg_namespace n ON n.oid = t.relnamespace
        WHERE n.nspname = 'inscada'
          AND i.relkind = 'i'
          AND NOT ix.indisprimary  -- Exclude primary key indexes
          AND NOT EXISTS (
            -- Exclude indexes that back constraints (unique, foreign key, etc.)
            SELECT 1 FROM pg_constraint c
            WHERE c.conindid = i.oid
          )
          AND EXISTS (
            SELECT 1 FROM pg_attribute a
            WHERE a.attrelid = t.oid
              AND a.attnum = ANY(ix.indkey)
              AND a.atttypid IN ('int4'::regtype, 'int8'::regtype)  -- Include both int4 and int8
          )
    LOOP
        EXECUTE format('DROP INDEX IF EXISTS inscada.%I;', r.index_name);
    END LOOP;
END $$;

-- Step 6: Change all primary key columns from serial/integer to text
DO $$
DECLARE
    r RECORD;
BEGIN
    FOR r IN
        SELECT c.table_name, c.column_name
        FROM information_schema.columns c
        JOIN information_schema.table_constraints tc
            ON c.table_name = tc.table_name
            AND c.table_schema = tc.table_schema
        JOIN information_schema.key_column_usage kcu
            ON tc.constraint_name = kcu.constraint_name
            AND tc.table_schema = kcu.table_schema
            AND c.column_name = kcu.column_name
        WHERE c.table_schema = 'inscada'
          AND tc.constraint_type = 'PRIMARY KEY'
          AND c.data_type IN ('integer', 'serial', 'bigint', 'bigserial')
          AND c.table_name != 'schema_version'  -- Exclude Flyway's schema_version table
    LOOP
        EXECUTE format('ALTER TABLE inscada.%I ALTER COLUMN %I TYPE text;', r.table_name, r.column_name);
        RAISE NOTICE 'Changed PK column %.% to text', r.table_name, r.column_name;
    END LOOP;
END $$;

-- Step 7: Change all other integer columns that were foreign keys or might be id-related
DO $$
DECLARE
    r RECORD;
BEGIN
    FOR r IN
        SELECT table_name, column_name
        FROM information_schema.columns
        WHERE table_schema = 'inscada'
          AND data_type IN ('integer', 'serial', 'bigint', 'bigserial')
          AND (column_name LIKE '%_id' OR column_name = 'id')
          AND table_name != 'schema_version'  -- Exclude Flyway's schema_version table
    LOOP
        EXECUTE format('ALTER TABLE inscada.%I ALTER COLUMN %I TYPE text;', r.table_name, r.column_name);
        RAISE NOTICE 'Changed column %.% to text', r.table_name, r.column_name;
    END LOOP;
END $$;

-- Step 8: Change all unique constraint columns from integer to text
DO $$
DECLARE
    r RECORD;
BEGIN
    FOR r IN
        SELECT DISTINCT c.table_name, c.column_name
        FROM information_schema.columns c
        JOIN information_schema.table_constraints tc
            ON c.table_name = tc.table_name
            AND c.table_schema = tc.table_schema
        JOIN information_schema.constraint_column_usage ccu
            ON tc.constraint_name = ccu.constraint_name
            AND tc.table_schema = ccu.table_schema
            AND c.column_name = ccu.column_name
        WHERE c.table_schema = 'inscada'
          AND tc.constraint_type = 'UNIQUE'
          AND c.data_type IN ('integer', 'serial', 'bigint', 'bigserial')
          AND c.table_name != 'schema_version'  -- Exclude Flyway's schema_version table
    LOOP
        EXECUTE format('ALTER TABLE inscada.%I ALTER COLUMN %I TYPE text;', r.table_name, r.column_name);
        RAISE NOTICE 'Changed unique column %.% to text', r.table_name, r.column_name;
    END LOOP;
END $$;

-- Step 9: Recreate foreign key constraints
DO $$
DECLARE
    r RECORD;
    fk_sql TEXT;
BEGIN
    FOR r IN SELECT * FROM temp_foreign_keys LOOP
        fk_sql := format('ALTER TABLE inscada.%I ADD CONSTRAINT %I FOREIGN KEY (%I) REFERENCES inscada.%I (%I)',
            r.table_name, r.constraint_name, r.column_name, r.foreign_table_name, r.foreign_column_name);

        -- Add ON DELETE and ON UPDATE rules if they exist
        IF r.delete_rule != 'NO ACTION' THEN
            fk_sql := fk_sql || ' ON DELETE ' || r.delete_rule;
        END IF;
        IF r.update_rule != 'NO ACTION' THEN
            fk_sql := fk_sql || ' ON UPDATE ' || r.update_rule;
        END IF;

        BEGIN
            EXECUTE fk_sql;
            RAISE NOTICE 'Recreated foreign key: %.%', r.table_name, r.constraint_name;
        EXCEPTION
            WHEN others THEN
                RAISE WARNING 'Failed to recreate foreign key %.%: %', r.table_name, r.constraint_name, SQLERRM;
        END;
    END LOOP;
END $$;

-- Step 10: Recreate the dropped indexes with updated column types
DO $$
DECLARE
    r RECORD;
    new_def TEXT;
BEGIN
    FOR r IN SELECT * FROM temp_index_definitions LOOP
        -- Replace the schema prefix and recreate the index
        new_def := replace(r.index_definition, 'CREATE INDEX', 'CREATE INDEX IF NOT EXISTS');
        BEGIN
            EXECUTE new_def;
            RAISE NOTICE 'Successfully recreated index: %', r.index_name;
        EXCEPTION
            WHEN others THEN
                -- Log the error but continue with other indexes
                RAISE WARNING 'Failed to recreate index %: %', r.index_name, SQLERRM;
        END;
    END LOOP;
END $$;

-- Step 11: Clean up temporary tables
DROP TABLE IF EXISTS temp_index_definitions;
DROP TABLE IF EXISTS temp_foreign_keys;

-- Step 12: Update any sequences or default values that might reference the old type
-- Remove any remaining default nextval() references
DO $$
DECLARE
    r RECORD;
BEGIN
    FOR r IN
        SELECT table_name, column_name
        FROM information_schema.columns
        WHERE table_schema = 'inscada'
          AND column_default LIKE 'nextval%'
    LOOP
        EXECUTE format('ALTER TABLE inscada.%I ALTER COLUMN %I DROP DEFAULT;', r.table_name, r.column_name);
        RAISE NOTICE 'Removed default from %.%', r.table_name, r.column_name;
    END LOOP;
END $$;

-- Migration completed successfully
-- All primary keys, foreign keys, and id-related columns have been converted to text
-- Sequences have been dropped and indexes have been recreated
