-- This migration adds a database-level "defense-in-depth" security measure.
-- It creates a trigger that prevents any application-side user from creating or modifying
-- an entity (user, role, or space) to have its 'is_system' flag set to TRUE.
-- This makes the 'is_system' flag effectively "write-once" at the database level,
-- meaning it can only be set during initial data migrations, not by the application.

-- Step 1: Create the trigger function that contains the security logic.
CREATE OR REPLACE FUNCTION enforce_system_entity_protection()
RETURNS TRIGGER
LANGUAGE plpgsql
AS $$
BEGIN
    -- Rule 1: Prevent any INSERT operation from creating a new system entity via the application.
    IF TG_OP = 'INSERT' AND NEW.is_system = TRUE THEN
        RAISE EXCEPTION 'Operation not permitted: Cannot create a new system entity. The is_system flag must be false.';
    END IF;

    -- For an UPDATE operation, we have more nuanced rules.
    IF TG_OP = 'UPDATE' THEN
        -- Rule 2: Prevent an existing entity from being changed into a system entity.
        IF OLD.is_system = FALSE AND NEW.is_system = TRUE THEN
            RAISE EXCEPTION 'Operation not permitted: Cannot change an entity to be a system entity.';
        END IF;
    END IF;

    -- If the check passes, allow the operation to proceed.
    RETURN NEW;
END;
$$;

-- Step 2: Apply the trigger to the 'users' table.
CREATE TRIGGER protect_system_users_trigger
BEFORE INSERT OR UPDATE ON inscada.users
FOR EACH ROW EXECUTE FUNCTION enforce_system_entity_protection();

-- Step 3: Apply the trigger to the 'roles' table.
CREATE TRIGGER protect_system_roles_trigger
BEFORE INSERT OR UPDATE ON inscada.roles
FOR EACH ROW EXECUTE FUNCTION enforce_system_entity_protection();

-- Step 4: Apply the trigger to the 'spaces' table.
CREATE TRIGGER protect_system_spaces_trigger
BEFORE INSERT OR UPDATE ON inscada.space
FOR EACH ROW EXECUTE FUNCTION enforce_system_entity_protection();