-- Migrate custom menu data to joined inheritance model with menus as parent table
-- This migration converts the unified custom_menu table into joined inheritance with menus

DO $$
DECLARE
    custom_menu_record RECORD;
    new_menu_id TEXT;
BEGIN
    -- Step 1: Create temporary mapping table for ID conversion
    CREATE TEMP TABLE custom_menu_id_map (
        old_id TEXT,
        new_id TEXT
    );

    -- Step 2: Generate new UUIDs for all custom menu entries and store mapping
    FOR custom_menu_record IN
        SELECT custom_menu_id, name, icon, target, position, content_type, content,
               space_id, insert_user, insert_dttm, version_user, version_dttm,
               menu_order, parent_id
        FROM custom_menu
        ORDER BY custom_menu_id
    LOOP
        -- Generate new UUID for menu_id
        SELECT gen_random_uuid()::TEXT INTO new_menu_id;

        -- Store the mapping
        INSERT INTO custom_menu_id_map(old_id, new_id)
        VALUES (custom_menu_record.custom_menu_id::TEXT, new_menu_id);

        -- Insert into menus table (parent table) with new UUID
        INSERT INTO menus (menu_id, name, insert_user, insert_dttm, version_user, version_dttm, space_id, is_global)
        VALUES (
            new_menu_id,
            custom_menu_record.name,
            COALESCE(custom_menu_record.insert_user, 'system'),
            COALESCE(custom_menu_record.insert_dttm, NOW()),
            COALESCE(custom_menu_record.version_user, 'system'),
            COALESCE(custom_menu_record.version_dttm, NOW()),
            custom_menu_record.space_id,
            FALSE -- Custom menus are space-specific
        );
    END LOOP;

END $$;

-- Step 3: Add new columns to custom_menu table for joined inheritance
ALTER TABLE custom_menu ADD COLUMN menu_id TEXT;
ALTER TABLE custom_menu ADD COLUMN new_parent_id TEXT;

-- Step 4: Update the new columns with mapped values
UPDATE custom_menu cm
SET menu_id = m.new_id
FROM custom_menu_id_map m
WHERE cm.custom_menu_id::TEXT = m.old_id;

-- Update parent relationships using the mapping
UPDATE custom_menu cm
SET new_parent_id = m1.new_id
FROM custom_menu_id_map m1
WHERE cm.parent_id::TEXT = m1.old_id
  AND cm.parent_id IS NOT NULL;

-- Step 5: Drop old columns and rename new ones
ALTER TABLE custom_menu DROP CONSTRAINT IF EXISTS custom_menu_parent_fk;
ALTER TABLE custom_menu DROP COLUMN custom_menu_id;
ALTER TABLE custom_menu DROP COLUMN parent_id;
ALTER TABLE custom_menu RENAME COLUMN new_parent_id TO parent_menu_id;

-- Step 6: Add constraints for joined inheritance
ALTER TABLE custom_menu ADD CONSTRAINT pk_custom_menu PRIMARY KEY (menu_id);
ALTER TABLE custom_menu ADD CONSTRAINT fk_custom_menu_menu
    FOREIGN KEY (menu_id) REFERENCES menus(menu_id);

-- Add foreign key constraint for parent relationships
ALTER TABLE custom_menu ADD CONSTRAINT fk_custom_menu_parent
    FOREIGN KEY (parent_menu_id) REFERENCES menus(menu_id);

-- Step 7: Drop inherited columns from custom_menu (now in parent menus table)
ALTER TABLE custom_menu
    DROP COLUMN space_id,
    DROP COLUMN name,
    DROP COLUMN insert_user,
    DROP COLUMN insert_dttm,
    DROP COLUMN version_user,
    DROP COLUMN version_dttm;

-- Step 8: Create indexes for performance
CREATE INDEX idx_custom_menu_menu_id ON custom_menu(menu_id);
CREATE INDEX idx_custom_menu_parent ON custom_menu(parent_menu_id);
