CREATE or replace FUNCTION global_search(
  space_id int,
  search_term text,
  comparator text
)
RETURNS table(schemaname text, tablename text, columnname text, columnvalue text, rowvalue text)
AS $$
DECLARE
  rowctid tid;
  query text;
  clauses text[];
  columns text[];
  pos int;
  positions int[];
BEGIN
  FOR schemaname,tablename IN
    -- select tables for which all columns are readable
    SELECT t.table_schema, t.table_name
      FROM information_schema.tables t
      JOIN information_schema.schemata s ON
	(s.schema_name=t.table_schema)
    WHERE t.table_schema='inscada'
      AND t.table_type='BASE TABLE'
      AND EXISTS (SELECT 1 FROM information_schema.table_privileges p
	WHERE p.table_name=t.table_name
	  AND p.table_schema=t.table_schema
	  AND p.privilege_type='SELECT'
      )
	  AND EXISTS (SELECT 1 from information_schema.columns c
	WHERE c.table_name=t.table_name
	 AND  c.table_schema=t.table_schema
     AND  c.column_name = 'space_id')
  LOOP
    --RAISE INFO '%', format('Searching globally in table: %I.%I', schemaname, tablename);

    -- Get lists of columns and per-column boolean expressions
    SELECT array_agg(column_name ORDER BY ordinal_position),
           array_agg(format('%s(cast(%I as text), %L)', cast(comparator as regproc), column_name, search_term)
	     ORDER BY ordinal_position)
      FROM information_schema.columns
      WHERE table_name=tablename
        AND table_schema=schemaname
    INTO columns, clauses;

    -- Main query to get each matching row and the ordinal positions of matching columns
    query := format('SELECT s.ctid, p from (SELECT ctid,'
		    'array_positions(array[%s],true) AS p FROM ONLY %I.%I WHERE space_id = %s) s'
		    ' WHERE cardinality(p)>0',
      array_to_string(clauses, ','), schemaname, tablename, space_id);

    FOR rowctid,positions IN EXECUTE query -- for each matching row
    LOOP
      FOREACH pos IN ARRAY positions -- for each matching field
      LOOP
	    columnname := columns[pos];

	    EXECUTE format('SELECT %I, cast(to_jsonb(t) as text) FROM %I.%I t WHERE ctid=''%s''',
	    columnname, schemaname, tablename, rowctid) INTO columnvalue, rowvalue;

	    --RAISE INFO '%', format('Found in %I.%I.%I at ctid %s', schemaname, tablename, columnname, rowctid);
	RETURN NEXT;
      END LOOP;
    END LOOP;
  END LOOP; -- for each table
END;
$$ language plpgsql;

insert into inscada.menus (menu_id, name) values (98, 'Search');
insert into inscada.permissions(permission_id, name) values (252, 'GLOBAL_SEARCH');
