-- =============================================================================
-- V251: Fix validate_variable_reference function for joined inheritance tables
-- =============================================================================
-- Description: Fix the validate_variable_reference function to properly handle
--              joined inheritance tables (like analog_alarm, digital_alarm)
--              that don't have deleted_dttm columns directly
-- Author: System
-- Date: 2025-09-08
-- =============================================================================

CREATE OR REPLACE FUNCTION validate_variable_reference()
RETURNS TRIGGER AS $$
DECLARE
    is_being_deleted BOOLEAN := FALSE;
    column_exists BOOLEAN := FALSE;
BEGIN
    -- Check if the table has a deleted_dttm column before trying to access OLD.deleted_dttm
    SELECT EXISTS (
        SELECT 1 FROM information_schema.columns
        WHERE table_schema = 'inscada'
        AND table_name = TG_TABLE_NAME
        AND column_name = 'deleted_dttm'
    ) INTO column_exists;

    -- Only check for soft deletion if the table has deleted_dttm column
    -- Use nested IF statements to avoid PostgreSQL evaluating OLD.deleted_dttm when column doesn't exist
    IF column_exists THEN
        IF TG_OP = 'UPDATE' AND OLD.deleted_dttm IS NULL AND NEW.deleted_dttm IS NOT NULL THEN
            is_being_deleted := TRUE;
        END IF;
    END IF;

    -- Skip validation if the record is being soft-deleted
    IF is_being_deleted THEN
        RETURN NEW;
    END IF;

    -- For alarm table
    IF TG_TABLE_NAME = 'alarm' THEN
        -- Check if columns exist before validating
        SELECT EXISTS (
            SELECT 1 FROM information_schema.columns
            WHERE table_schema = 'inscada' AND table_name = 'alarm' AND column_name = 'off_time_variable_id'
        ) INTO column_exists;

        IF column_exists AND NEW.off_time_variable_id IS NOT NULL AND NOT inscada.check_variable_exists(NEW.off_time_variable_id) THEN
            RAISE EXCEPTION 'Variable with id % does not exist', NEW.off_time_variable_id;
        END IF;

        SELECT EXISTS (
            SELECT 1 FROM information_schema.columns
            WHERE table_schema = 'inscada' AND table_name = 'alarm' AND column_name = 'on_time_variable_id'
        ) INTO column_exists;

        IF column_exists AND NEW.on_time_variable_id IS NOT NULL AND NOT inscada.check_variable_exists(NEW.on_time_variable_id) THEN
            RAISE EXCEPTION 'Variable with id % does not exist', NEW.on_time_variable_id;
        END IF;
    END IF;

    -- For digital_alarm
    IF TG_TABLE_NAME = 'digital_alarm' THEN
        SELECT EXISTS (
            SELECT 1 FROM information_schema.columns
            WHERE table_schema = 'inscada' AND table_name = 'digital_alarm' AND column_name = 'variable_a_id'
        ) INTO column_exists;

        IF column_exists AND NEW.variable_a_id IS NOT NULL AND NOT inscada.check_variable_exists(NEW.variable_a_id) THEN
            RAISE EXCEPTION 'Variable with id % does not exist', NEW.variable_a_id;
        END IF;

        SELECT EXISTS (
            SELECT 1 FROM information_schema.columns
            WHERE table_schema = 'inscada' AND table_name = 'digital_alarm' AND column_name = 'variable_b_id'
        ) INTO column_exists;

        IF column_exists AND NEW.variable_b_id IS NOT NULL AND NOT inscada.check_variable_exists(NEW.variable_b_id) THEN
            RAISE EXCEPTION 'Variable with id % does not exist', NEW.variable_b_id;
        END IF;
    END IF;

    -- For analog_alarm
    IF TG_TABLE_NAME = 'analog_alarm' THEN
        SELECT EXISTS (
            SELECT 1 FROM information_schema.columns
            WHERE table_schema = 'inscada' AND table_name = 'analog_alarm' AND column_name = 'variable_id'
        ) INTO column_exists;

        IF column_exists AND NEW.variable_id IS NOT NULL AND NOT inscada.check_variable_exists(NEW.variable_id) THEN
            RAISE EXCEPTION 'Variable with id % does not exist', NEW.variable_id;
        END IF;
    END IF;

    -- For data_transfer_detail
    IF TG_TABLE_NAME = 'data_transfer_detail' THEN
        SELECT EXISTS (
            SELECT 1 FROM information_schema.columns
            WHERE table_schema = 'inscada' AND table_name = 'data_transfer_detail' AND column_name = 'source_var_id'
        ) INTO column_exists;

        IF column_exists AND NEW.source_var_id IS NOT NULL AND NOT inscada.check_variable_exists(NEW.source_var_id) THEN
            RAISE EXCEPTION 'Variable with id % does not exist', NEW.source_var_id;
        END IF;

        SELECT EXISTS (
            SELECT 1 FROM information_schema.columns
            WHERE table_schema = 'inscada' AND table_name = 'data_transfer_detail' AND column_name = 'target_var_id'
        ) INTO column_exists;

        IF column_exists AND NEW.target_var_id IS NOT NULL AND NOT inscada.check_variable_exists(NEW.target_var_id) THEN
            RAISE EXCEPTION 'Variable with id % does not exist', NEW.target_var_id;
        END IF;
    END IF;

    -- For report_variable
    IF TG_TABLE_NAME = 'report_variable' THEN
        SELECT EXISTS (
            SELECT 1 FROM information_schema.columns
            WHERE table_schema = 'inscada' AND table_name = 'report_variable' AND column_name = 'deviation_variable_id'
        ) INTO column_exists;

        IF column_exists AND NEW.deviation_variable_id IS NOT NULL AND NOT inscada.check_variable_exists(NEW.deviation_variable_id) THEN
            RAISE EXCEPTION 'Variable with id % does not exist', NEW.deviation_variable_id;
        END IF;

        SELECT EXISTS (
            SELECT 1 FROM information_schema.columns
            WHERE table_schema = 'inscada' AND table_name = 'report_variable' AND column_name = 'total_variable_id'
        ) INTO column_exists;

        IF column_exists AND NEW.total_variable_id IS NOT NULL AND NOT inscada.check_variable_exists(NEW.total_variable_id) THEN
            RAISE EXCEPTION 'Variable with id % does not exist', NEW.total_variable_id;
        END IF;

        SELECT EXISTS (
            SELECT 1 FROM information_schema.columns
            WHERE table_schema = 'inscada' AND table_name = 'report_variable' AND column_name = 'variable_id'
        ) INTO column_exists;

        IF column_exists AND NEW.variable_id IS NOT NULL AND NOT inscada.check_variable_exists(NEW.variable_id) THEN
            RAISE EXCEPTION 'Variable with id % does not exist', NEW.variable_id;
        END IF;
    END IF;

    -- For trend_tag
    IF TG_TABLE_NAME = 'trend_tag' THEN
        SELECT EXISTS (
            SELECT 1 FROM information_schema.columns
            WHERE table_schema = 'inscada' AND table_name = 'trend_tag' AND column_name = 'variable_id'
        ) INTO column_exists;

        IF column_exists AND NEW.variable_id IS NOT NULL AND NOT inscada.check_variable_exists(NEW.variable_id) THEN
            RAISE EXCEPTION 'Variable with id % does not exist', NEW.variable_id;
        END IF;
    END IF;

    -- For map_variable
    IF TG_TABLE_NAME = 'map_variable' THEN
        SELECT EXISTS (
            SELECT 1 FROM information_schema.columns
            WHERE table_schema = 'inscada' AND table_name = 'map_variable' AND column_name = 'variable_id'
        ) INTO column_exists;

        IF column_exists AND NEW.variable_id IS NOT NULL AND NOT inscada.check_variable_exists(NEW.variable_id) THEN
            RAISE EXCEPTION 'Variable with id % does not exist', NEW.variable_id;
        END IF;
    END IF;

    -- For monitor_variable
    IF TG_TABLE_NAME = 'monitor_variable' THEN
        SELECT EXISTS (
            SELECT 1 FROM information_schema.columns
            WHERE table_schema = 'inscada' AND table_name = 'monitor_variable' AND column_name = 'variable_id'
        ) INTO column_exists;

        IF column_exists AND NEW.variable_id IS NOT NULL AND NOT inscada.check_variable_exists(NEW.variable_id) THEN
            RAISE EXCEPTION 'Variable with id % does not exist', NEW.variable_id;
        END IF;
    END IF;

    RETURN NEW;
END;
$$ LANGUAGE plpgsql;
