-- This migration script handles the transition to case-insensitive role names.
-- It safely resolves any existing duplicate roles (e.g., 'Admin' and 'admin') by
-- renaming the duplicates with a unique suffix (e.g., 'admin-a4e1').
-- It then creates a new, case-insensitive unique index that respects soft deletes.

-- Step 1: Find all roles that are case-insensitive duplicates within the same space
-- and update their names with a unique suffix.
WITH ranked_roles AS (
    SELECT
        role_id,
        name,
        -- Rank roles within each duplicate group. The one with the earliest creation date gets rank 1.
        ROW_NUMBER() OVER(
            PARTITION BY LOWER(name), space_id
            ORDER BY insert_dttm, role_id
        ) as rn
    FROM
        inscada.roles
    WHERE
        deleted_dttm IS NULL
),
roles_to_rename AS (
    -- Select all roles that are not the first one in their duplicate group (rn > 1).
    -- These are the ones we need to rename.
    SELECT
        role_id,
        -- Append a short, unique suffix from the role's own ID for deterministic renaming,
        -- matching the strategy used in the V267 username migration.
        name || '-' || SUBSTRING(REPLACE(role_id::text, '-', ''), 1, 4) as new_name
    FROM
        ranked_roles
    WHERE
        rn > 1
)
UPDATE
    inscada.roles r
SET
    name = rtr.new_name
FROM
    roles_to_rename rtr
WHERE
    r.role_id = rtr.role_id;

-- Step 2: Drop the old case-sensitive unique index.
DROP INDEX IF EXISTS inscada.roles_name_space_uindex;

-- Step 3: Create a new, case-insensitive unique index that only applies to non-deleted rows.
-- This is a partial index, which is a powerful feature of PostgreSQL.
CREATE UNIQUE INDEX IF NOT EXISTS roles_lower_name_space_id_unique_idx
ON inscada.roles (LOWER(name), space_id)
WHERE deleted_dttm IS NULL;