-- Migration to fix the validate_variable_reference function for proper soft delete compatibility
-- This addresses the issue where triggers from V228 were calling validate_variable_reference()
-- but V240 renamed it to is_variable_referenced(), causing database errors during project deletion

-- Update the validate_variable_reference function to properly handle soft deletion
-- This function is called by triggers created in V228 for various tables

-- Drop the incorrectly named function from V240 if it exists
DROP FUNCTION IF EXISTS is_variable_referenced(VARCHAR);

CREATE OR REPLACE FUNCTION validate_variable_reference()
RETURNS TRIGGER AS $$
DECLARE
    is_being_deleted BOOLEAN := FALSE;
BEGIN
    -- Check if the record is being soft-deleted (deleted_dttm is being set to a non-null value)
    IF TG_OP = 'UPDATE' AND OLD.deleted_dttm IS NULL AND NEW.deleted_dttm IS NOT NULL THEN
        is_being_deleted := TRUE;
    END IF;

    -- Skip validation if the record is being soft-deleted
    IF is_being_deleted THEN
        RETURN NEW;
    END IF;

    -- For alarm table
    IF TG_TABLE_NAME = 'alarm' THEN
        IF NEW.off_time_variable_id IS NOT NULL AND NOT inscada.check_variable_exists(NEW.off_time_variable_id) THEN
            RAISE EXCEPTION 'Variable with id % does not exist', NEW.off_time_variable_id;
        END IF;

        IF NEW.on_time_variable_id IS NOT NULL AND NOT inscada.check_variable_exists(NEW.on_time_variable_id) THEN
            RAISE EXCEPTION 'Variable with id % does not exist', NEW.on_time_variable_id;
        END IF;
    END IF;

    -- For digital_alarm
    IF TG_TABLE_NAME = 'digital_alarm' THEN
        IF NEW.variable_a_id IS NOT NULL AND NOT inscada.check_variable_exists(NEW.variable_a_id) THEN
            RAISE EXCEPTION 'Variable with id % does not exist', NEW.variable_a_id;
        END IF;

        IF NEW.variable_b_id IS NOT NULL AND NOT inscada.check_variable_exists(NEW.variable_b_id) THEN
            RAISE EXCEPTION 'Variable with id % does not exist', NEW.variable_b_id;
        END IF;
    END IF;

    -- For analog_alarm
    IF TG_TABLE_NAME = 'analog_alarm' THEN
        IF NEW.variable_id IS NOT NULL AND NOT inscada.check_variable_exists(NEW.variable_id) THEN
            RAISE EXCEPTION 'Variable with id % does not exist', NEW.variable_id;
        END IF;
    END IF;

    -- For data_transfer_detail
    IF TG_TABLE_NAME = 'data_transfer_detail' THEN
        IF NEW.source_var_id IS NOT NULL AND NOT inscada.check_variable_exists(NEW.source_var_id) THEN
            RAISE EXCEPTION 'Variable with id % does not exist', NEW.source_var_id;
        END IF;

        IF NEW.target_var_id IS NOT NULL AND NOT inscada.check_variable_exists(NEW.target_var_id) THEN
            RAISE EXCEPTION 'Variable with id % does not exist', NEW.target_var_id;
        END IF;
    END IF;

    -- For report_variable
    IF TG_TABLE_NAME = 'report_variable' THEN
        IF NEW.deviation_variable_id IS NOT NULL AND NOT inscada.check_variable_exists(NEW.deviation_variable_id) THEN
            RAISE EXCEPTION 'Variable with id % does not exist', NEW.deviation_variable_id;
        END IF;

        IF NEW.total_variable_id IS NOT NULL AND NOT inscada.check_variable_exists(NEW.total_variable_id) THEN
            RAISE EXCEPTION 'Variable with id % does not exist', NEW.total_variable_id;
        END IF;

        IF NEW.variable_id IS NOT NULL AND NOT inscada.check_variable_exists(NEW.variable_id) THEN
            RAISE EXCEPTION 'Variable with id % does not exist', NEW.variable_id;
        END IF;
    END IF;

    -- For trend_tag
    IF TG_TABLE_NAME = 'trend_tag' THEN
        IF NEW.variable_id IS NOT NULL AND NOT inscada.check_variable_exists(NEW.variable_id) THEN
            RAISE EXCEPTION 'Variable with id % does not exist', NEW.variable_id;
        END IF;
    END IF;

    -- For map_variable
    IF TG_TABLE_NAME = 'map_variable' THEN
        IF NEW.variable_id IS NOT NULL AND NOT inscada.check_variable_exists(NEW.variable_id) THEN
            RAISE EXCEPTION 'Variable with id % does not exist', NEW.variable_id;
        END IF;
    END IF;

    -- For monitor_variable
    IF TG_TABLE_NAME = 'monitor_variable' THEN
        IF NEW.variable_id IS NOT NULL AND NOT inscada.check_variable_exists(NEW.variable_id) THEN
            RAISE EXCEPTION 'Variable with id % does not exist', NEW.variable_id;
        END IF;
    END IF;

    RETURN NEW;
END;
$$ LANGUAGE plpgsql;