-- Migrate existing roles to create space-specific instances
-- This migration creates space-specific copies of all existing roles and their permissions
-- Note: This ignores the is_global flag and creates space instances for all permissions/roles

DO $$
DECLARE
    current_timestamp TIMESTAMPTZ := NOW();
BEGIN

    -- Step 1: Bulk create space-specific permissions (ULTRA FAST WITH CROSS JOIN)
    INSERT INTO permissions (permission_id, name, is_global, space_id, insert_user, insert_dttm, version_user, version_dttm)
    SELECT
        gen_random_uuid()::text,
        p.name,
        p.is_global,
        s.space_id,
        'system',
        current_timestamp,
        'system',
        current_timestamp
    FROM permissions p
    CROSS JOIN space s
    LEFT JOIN permissions existing_p ON existing_p.name = p.name AND existing_p.space_id = s.space_id
    WHERE p.space_id IS NULL
      AND existing_p.permission_id IS NULL;

    -- Step 1b: Bulk create space-specific menus (ULTRA FAST WITH CROSS JOIN)
    INSERT INTO menus (menu_id, name, is_global, space_id, insert_user, insert_dttm, version_user, version_dttm)
    SELECT
        gen_random_uuid()::text,
        m.name,
        m.is_global,
        s.space_id,
        'system',
        current_timestamp,
        'system',
        current_timestamp
    FROM menus m
    CROSS JOIN space s
    LEFT JOIN menus existing_m ON existing_m.name = m.name AND existing_m.space_id = s.space_id
    WHERE m.space_id IS NULL
      AND existing_m.menu_id IS NULL;

    -- Step 2: Bulk create space-specific roles (ULTRA FAST WITH CROSS JOIN)
    INSERT INTO roles (role_id, name, space_id, insert_user, insert_dttm, version_user, version_dttm)
    SELECT
        gen_random_uuid()::text,
        r.name,
        s.space_id,
        COALESCE(r.insert_user, 'system'),
        COALESCE(r.insert_dttm, current_timestamp),
        COALESCE(r.version_user, 'system'),
        COALESCE(r.version_dttm, current_timestamp)
    FROM roles r
    CROSS JOIN space s
    WHERE r.space_id IS NULL;

    -- Step 3: Bulk copy role-permission relationships (OPTIMIZED)
    INSERT INTO role_permissions (role_id, permission_id)
    SELECT DISTINCT
        new_r.role_id,
        new_p.permission_id
    FROM role_permissions rp
    JOIN roles orig_r ON rp.role_id = orig_r.role_id AND orig_r.space_id IS NULL
    JOIN permissions orig_p ON rp.permission_id = orig_p.permission_id AND orig_p.space_id IS NULL
    JOIN roles new_r ON new_r.name = orig_r.name AND new_r.space_id IS NOT NULL
    JOIN permissions new_p ON new_p.name = orig_p.name AND new_p.space_id = new_r.space_id; -- Ensures role and permission are in the same space

    -- Step 3b: Bulk copy role-member relationships (OPTIMIZED)
    INSERT INTO role_members (role_id, user_id)
    SELECT DISTINCT
        new_r.role_id,
        rm.user_id
    FROM role_members rm
    JOIN roles orig_r ON rm.role_id = orig_r.role_id AND orig_r.space_id IS NULL
    JOIN roles new_r ON new_r.name = orig_r.name AND new_r.space_id IS NOT NULL
    -- Ensure the user is actually a member of the target space before adding them to the role
    JOIN space_users su ON rm.user_id = su.user_id AND su.space_id = new_r.space_id;

    -- Step 4: Bulk copy role-menu relationships (OPTIMIZED)
    INSERT INTO role_menus (role_id, menu_id)
    SELECT DISTINCT
        new_r.role_id,
        new_m.menu_id
    FROM role_menus rm
    JOIN roles orig_r ON rm.role_id = orig_r.role_id AND orig_r.space_id IS NULL
    JOIN menus orig_m ON rm.menu_id = orig_m.menu_id AND orig_m.space_id IS NULL
    JOIN roles new_r ON new_r.name = orig_r.name AND new_r.space_id IS NOT NULL
    JOIN menus new_m ON new_m.name = orig_m.name AND new_m.space_id = new_r.space_id; -- Ensures role and menu are in the same space


    -- Step 5: Clean up - Remove original template roles but keep template permissions
    -- Remove original role-permission relationships for template roles
    DELETE FROM role_permissions
    WHERE role_id IN (SELECT role_id FROM roles WHERE space_id IS NULL);

    -- Remove role-menu relationships for template roles
    DELETE FROM role_menus
    WHERE role_id IN (SELECT role_id FROM roles WHERE space_id IS NULL);

    -- Remove role-member relationships for template roles
    DELETE FROM role_members
    WHERE role_id IN (SELECT role_id FROM roles WHERE space_id IS NULL);

    -- Remove template roles (space_id = NULL)
    DELETE FROM roles WHERE space_id IS NULL;

    -- Keep template permissions for future use when creating new spaces
    -- DELETE FROM permissions WHERE space_id IS NULL; -- COMMENTED OUT - keeping templates

    -- Keep template menus for future use when creating new spaces
    -- DELETE FROM menus WHERE space_id IS NULL; -- COMMENTED OUT - keeping templates

END $$;

-- Add indexes for future queries
CREATE INDEX IF NOT EXISTS idx_role_members_role_id_user_id ON role_members (role_id, user_id);
CREATE INDEX IF NOT EXISTS idx_role_menus_role_id_menu_id ON role_menus (role_id, menu_id);
CREATE INDEX IF NOT EXISTS idx_role_permissions_role_id_permission_id ON role_permissions (role_id, permission_id);
CREATE INDEX IF NOT EXISTS idx_space_users_space_id_user_id ON space_users (space_id, user_id);
CREATE INDEX IF NOT EXISTS idx_space_users_user_id ON space_users (user_id);