-- Migration to update unique indexes and constraints to include deleted_dttm for soft delete support
-- This ensures that unique constraints only apply to non-deleted records

DO $$
DECLARE
    index_record RECORD;
    constraint_record RECORD;
    drop_statement TEXT;
    create_statement TEXT;
    columns_list TEXT;
    table_name_var TEXT;
    index_name_var TEXT;
    constraint_name_var TEXT;
BEGIN
    -- First, handle unique constraints (not indexes)
    FOR constraint_record IN
        SELECT
            tc.table_name,
            tc.constraint_name,
            string_agg(kcu.column_name, ', ' ORDER BY kcu.ordinal_position) AS columns
        FROM information_schema.table_constraints tc
        JOIN information_schema.key_column_usage kcu
            ON tc.constraint_name = kcu.constraint_name
            AND tc.table_schema = kcu.table_schema
        WHERE tc.table_schema = 'inscada'
          AND tc.constraint_type = 'UNIQUE'
          -- Only process tables that have deleted_dttm column
          AND EXISTS (
              SELECT 1
              FROM information_schema.columns col
              WHERE col.table_schema = 'inscada'
                AND col.table_name = tc.table_name
                AND col.column_name = 'deleted_dttm'
          )
          -- Exclude primary key constraints
          AND NOT EXISTS (
              SELECT 1
              FROM information_schema.table_constraints tc2
              WHERE tc2.table_schema = 'inscada'
                AND tc2.table_name = tc.table_name
                AND tc2.constraint_name = tc.constraint_name
                AND tc2.constraint_type = 'PRIMARY KEY'
          )
        GROUP BY tc.table_name, tc.constraint_name
        ORDER BY tc.table_name, tc.constraint_name
    LOOP
        table_name_var := constraint_record.table_name;
        constraint_name_var := constraint_record.constraint_name;
        columns_list := constraint_record.columns;

        -- Drop the unique constraint
        drop_statement := 'ALTER TABLE inscada.' || table_name_var || ' DROP CONSTRAINT IF EXISTS ' || constraint_name_var || ';';
        EXECUTE drop_statement;
        RAISE NOTICE 'Dropped unique constraint: % on table: %', constraint_name_var, table_name_var;

        -- Create new unique index with deleted_dttm included and WHERE clause
        create_statement := 'CREATE UNIQUE INDEX ' || constraint_name_var ||
                           ' ON inscada.' || table_name_var ||
                           ' (' || columns_list || ') ' ||
                           ' WHERE deleted_dttm IS NULL;';

        EXECUTE create_statement;
        RAISE NOTICE 'Created updated unique index: % on table: % with columns: % WHERE deleted_dttm IS NULL',
                     constraint_name_var, table_name_var, columns_list;
    END LOOP;

    -- Then, handle unique indexes that are not backing constraints
    FOR index_record IN
        SELECT
            n.nspname AS schema_name,
            t.relname AS table_name,
            i.relname AS index_name,
            array_to_string(array_agg(a.attname ORDER BY c.ordinality), ', ') AS columns
        FROM pg_class t
        JOIN pg_namespace n ON n.oid = t.relnamespace
        JOIN pg_index ix ON t.oid = ix.indrelid
        JOIN pg_class i ON i.oid = ix.indexrelid
        JOIN unnest(ix.indkey) WITH ORDINALITY AS c(attnum, ordinality) ON true
        JOIN pg_attribute a ON a.attrelid = t.oid AND a.attnum = c.attnum
        WHERE n.nspname = 'inscada'
          AND ix.indisunique = true
          AND ix.indisprimary = false
          -- Only process tables that have deleted_dttm column
          AND EXISTS (
              SELECT 1
              FROM information_schema.columns col
              WHERE col.table_schema = 'inscada'
                AND col.table_name = t.relname
                AND col.column_name = 'deleted_dttm'
          )
          -- Exclude indexes that are backing constraints (already handled above)
          AND NOT EXISTS (
              SELECT 1
              FROM information_schema.table_constraints tc
              WHERE tc.table_schema = 'inscada'
                AND tc.table_name = t.relname
                AND tc.constraint_name = i.relname
                AND tc.constraint_type = 'UNIQUE'
          )
        GROUP BY n.nspname, t.relname, i.relname
        ORDER BY t.relname, i.relname
    LOOP
        table_name_var := index_record.table_name;
        index_name_var := index_record.index_name;
        columns_list := index_record.columns;

        -- Drop the existing unique index
        drop_statement := 'DROP INDEX IF EXISTS inscada.' || index_name_var || ';';
        EXECUTE drop_statement;
        RAISE NOTICE 'Dropped unique index: % on table: %', index_name_var, table_name_var;

        -- Create new unique index with deleted_dttm included and WHERE clause
        create_statement := 'CREATE UNIQUE INDEX ' || index_name_var ||
                           ' ON inscada.' || table_name_var ||
                           ' (' || columns_list || ') ' ||
                           ' WHERE deleted_dttm IS NULL;';

        EXECUTE create_statement;
        RAISE NOTICE 'Created updated unique index: % on table: % with columns: % WHERE deleted_dttm IS NULL',
                     index_name_var, table_name_var, columns_list;

    END LOOP;

    RAISE NOTICE 'Updated all unique indexes and constraints in inscada schema to support soft delete';
END $$;
